package com.huigou.topsun.product.application.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ObjectUtil;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.huigou.cache.SystemCache;
import com.huigou.data.query.model.QueryDescriptor;
import com.huigou.data.query.model.QueryModel;
import com.huigou.topsun.product.application.*;
import com.huigou.topsun.product.domain.*;
import com.huigou.topsun.product.domain.query.ProductQueryRequest;
import com.huigou.topsun.product.domain.vo.ProductAllDataVo;
import com.huigou.topsun.product.domain.vo.ProductReturn;
import com.huigou.topsun.product.repository.*;
import com.huigou.topsun.sap.common.HttpClient;
import com.huigou.topsun.util.EntityConvertUtil;
import com.huigou.topsun.util.PersistTool;
import com.huigou.uasp.bmp.common.application.BaseApplication;
import com.huigou.util.StringUtil;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.TypedQuery;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

// @RequiredArgsConstructor 通过构造方法注入同 ProductApplication(Class a) { this.a = a; }

/**
 * description 针对表【product(产品)】的数据库操作Service实现
 *
 * @author 16508
 * @createDate 2023-11-22 10:24:31
 */
@Service
@RequiredArgsConstructor
public class ProductApplicationImpl extends BaseApplication implements ProductApplication {

    @PersistenceContext(unitName = "system")
    private EntityManager entityManager;

    private final PersistTool persist;

//    private final EntityManagerFactory entityManagerFactory;

    private final ObjectMapper objectMapper;
    private final ProductRepository productRepository;
    private final ProductDetailRepository detailRepository;

    // 类型

    private final BrandRepository brandRepository;
    private final FactoryRepository factoryRepository;
    private final ColorRepository colorRepository;
    private final ProductCategoryRepository categoryRepository;

    // 详细信息

    private final ProductTypesetConfRepository typesetConfRepository;
    private final ProductLookedRepository lookedRepository;
    private final ProductPublishedConfRepository publishedConfRepository;
    private final ProductLossRepository lossRepository;
    private final ProductMaterialRepository materialRepository;

    private final ProductFaceRepository faceRepository;
    private final ProductFaceColorRepository faceColorRepository;
    private final ProductTechnologyRepository technologyRepository;
    private final ProductProcessRepository processRepository;

    private final ProductDetailApplication productDetailApplication;
    private final ProductFaceApplication productFaceApplication;
    private final ProductLookedApplication productLookedApplication;
    private final ProductPublishedConfApplication productPublishedConfApplication;
    private final ProductTechnologyApplication productTechnologyApplication;
    private final ProductLossApplication productLossApplication;
    private final ProductProcessApplication productProcessApplication;
    private final ProductFaceColorApplication productFaceColorApplication;
    private final ProductMaterialApplication productMaterialApplication;

    @Autowired
    private HttpClient httpClient;

    @Override
    public ArrayList<Map<String, Object>> findProductPage(int page, int size) {
        ArrayList<Product> productPage = (ArrayList<Product>) productRepository.findAll();
        return objectMapper.convertValue(productPage, new TypeReference<ArrayList<Map<String, Object>>>() {
        });
    }

    /**
     * description 查询产品所有详细信息
     *
     * @param product 产品ID
     * @return java.util.Map<java.lang.String, java.lang.Object>
     * @author qinzhenguan
     * @createDate 2023/11/29 13:19
     */
    @Override
    public Map<String, Object> queryDetailAll(Product product) {
        Map<String, Object> resultMap = new HashMap<>(200);
        String productId = product.getProductId();
        product = productRepository.findOne(productId);
        ProductDetail productDetail = detailRepository.findByProductId(productId);
        ProductTechnology productTechnology = technologyRepository.findByProductId(productId);

        // ---------------------------------  类别查询 ---------------------------------
        ArrayList<Brand> brands = (ArrayList<Brand>) brandRepository.findAll();
        ArrayList<Factory> factorys = (ArrayList<Factory>) factoryRepository.findAll();
        ArrayList<Color> colors = (ArrayList<Color>) colorRepository.findAll();
        ArrayList<ProductCategory> categorys = (ArrayList<ProductCategory>) categoryRepository.findAll();

        // ---------------------------------  详情查询 ---------------------------------

        ArrayList<ProductMaterial> material = (ArrayList<ProductMaterial>) materialRepository.findByProductId(productId);
        ArrayList<ProductLoss> loss = (ArrayList<ProductLoss>) lossRepository.findByProductId(productId);
        Set<Object> entitys = new HashSet<>();
        entitys.add(typesetConfRepository.findByProductId(productId));
        entitys.add(lookedRepository.findByProductId(productId));
        entitys.add(publishedConfRepository.findByProductId(productId));
        entitys.add(productDetail);
        entitys.add(productTechnology == null ? new ProductTechnology() : productTechnology);
        entitys.add(product);
        entitys.add(material);

        Map<String, Object> assemble = EntityConvertUtil.assembleResultOfForm(entitys);
        resultMap.put("rawData", assemble);
        assemble.put("productLoss", loss);

        // ---------------------------------  版面查询 (关联产品id)---------------------------------
        if (product != null) {
            //ArrayList<ProductFace> frontFaces = new ArrayList<>();
            if (product.getProductId() != null) {
                //ProductFace frontFace = faceRepository.findByProductFaceId(productDetail.getRightProductFaceId());
                //frontFace.setBack(false);
                //ArrayList <ProductFaceColor> frontColors = (ArrayList<ProductFaceColor>) faceColorRepository.findByProductFaceId(productDetail.getRightProductFaceId());
                List<ProductFaceColor> frontColors = faceColorRepository.findByProductIdAndFace(productId, "front");
                frontColors = this.getColor(frontColors);
                List<ProductFaceColor> backColors = faceColorRepository.findByProductIdAndFace(productId, "back");
                backColors = this.getColor(backColors);
                //frontFaces.add(frontFace);
                assemble.put("frontColors", frontColors == null ? new ArrayList<>() : frontColors);
                assemble.put("backColors", backColors == null ? new ArrayList<>() : backColors);
            } else {
                ProductFaceColor fc = new ProductFaceColor();
                fc.setProductFaceColorId("111");
                fc.setColorId("1");
                ArrayList<ProductFaceColor> frontColors = new ArrayList<>();
                frontColors.add(fc);
                assemble.put("frontColors", frontColors);
            }
            /*if(productDetail.getBackProductFaceId() != null) {
                ProductFace backFace = faceRepository.findByProductFaceId(productDetail.getBackProductFaceId());
                backFace.setBack(true);
                ArrayList <ProductFaceColor> backColors =
                        (ArrayList<ProductFaceColor>) faceColorRepository.findByProductFaceId(productDetail.getBackProductFaceId());
                //frontFaces.add(backFace);
                assemble.put("backColors", backColors == null ? new ArrayList<>() : backColors);
            } else {
                ProductFaceColor fc = new ProductFaceColor();
                fc.setProductFaceColorId("111");
                fc.setColorId("1");
                ArrayList<ProductFaceColor> frontColors =  new ArrayList<>();
                frontColors.add(fc);
                assemble.put("backColors", frontColors);
            }*/
            //assemble.put("frontFaces", frontFaces);

        }

        // ---------------------------------  工序查询 ---------------------------------
        if (productTechnology != null) {
            if (productTechnology.getProductTechnologyId() != null) {
                ArrayList<ProductProcess> processesed =
                        (ArrayList<ProductProcess>) processRepository.findByProductTechnologyId(productTechnology.getProductTechnologyId());
                assemble.put("processed", processesed == null ? new ArrayList<>() : processesed);
            } else {
                ProductProcess fc = new ProductProcess();
                fc.setProductProcessId("111");
                fc.setProductProcessId("1");
                ArrayList<ProductProcess> processed = new ArrayList<>();
                processed.add(fc);
                assemble.put("processed", processed);
            }
        }

        HashMap<String, Object> listMaps = new HashMap<>(100);
        listMaps.put("selectedBrand", objectMapper.convertValue(brands, List.class));
        listMaps.put("selectedFactory", objectMapper.convertValue(factorys, List.class));
        listMaps.put("selectedColor", objectMapper.convertValue(colors, List.class));
        listMaps.put("selectedCategory", objectMapper.convertValue(categorys, List.class));
        resultMap.put("listMaps", listMaps);
        return resultMap;
    }

    public List<ProductFaceColor> getColor(List<ProductFaceColor> colorList) {
        colorList.forEach(frontColor -> {
            Color color = colorRepository.findOne(frontColor.getColorId());
            frontColor.setColorName(color.getColorName());
        });
        return colorList;
    }

    /***
     * description  保存所有产品关联信息
     * @param newData   发生变更的所有数据实体，传入该方法所有数据都将持久化。
     * @return java.util.HashMap<java.lang.String, java.lang.Object>
     * @author qinzhenguan
     * @createDate 2023/12/26 17:05
     */
    @Override
    @Transactional(transactionManager = "transactionManager")
    public HashMap<String, Object> saveOrUpdataOnAllDetail(HashMap<String, Object> newData) {
        //如果productId有值，则是修改保存。否则是新增保存
        String productId = (String) newData.get("productId");

        //先保存产品主表，获取到产品id，关联其他属性表
        Product product = objectMapper.convertValue(newData.get("Product"), Product.class);
        if (ObjectUtil.isNotEmpty(product)) {
            persist.persistEntity(product);
            //再次赋值，确保productId有值
            productId = product.getProductId();
        }
        //产品详情
        ProductDetail productDetail = objectMapper.convertValue(newData.get("ProductDetail"), ProductDetail.class);
        if (ObjectUtil.isNotEmpty(productDetail)) {
            if (StringUtil.isBlank(productDetail.getProductId())) {
                productDetail.setProductId(productId);
            }
        }
        //产品外观
        ProductLooked productLooked = objectMapper.convertValue(newData.get("ProductLooked"), ProductLooked.class);
        if (ObjectUtil.isNotEmpty(productLooked)) {
            if (StringUtil.isBlank(productLooked.getProductId())) {
                productLooked.setProductId(productId);
            }
        }
        //产品正面
        ProductFace frontFace = objectMapper.convertValue(newData.get("frontFace"), ProductFace.class);
        if (ObjectUtil.isNotEmpty(frontFace)) {
            persist.persistEntity(frontFace);
        }
        //产品反面
        ProductFace backFace = objectMapper.convertValue(newData.get("backFace"), ProductFace.class);
        if (ObjectUtil.isNotEmpty(backFace)) {
            persist.persistEntity(backFace);
            //this.convertToFaceColors(uncheckedCast(newData.get("backColors")),backFace.getProductFaceId());
        }
        //产品工艺
        ProductTechnology productTechnology = objectMapper.convertValue(newData.get("ProductTechnology"), ProductTechnology.class);
        if (ObjectUtil.isNotEmpty(productTechnology)) {
            if (StringUtil.isBlank(productTechnology.getProductId())) {
                productTechnology.setProductId(productId);
            }
            persist.persistEntity(productTechnology);
            convertToProcesseds(uncheckedCast(newData.get("processed")), productTechnology.getProductTechnologyId());
        }

        //产品参数配置
        ProductPublishedConf productPublishedConf = objectMapper.convertValue(newData.get("ProductPublishedConf"), ProductPublishedConf.class);
        if (ObjectUtil.isNotEmpty(productPublishedConf)) {
            if (StringUtil.isBlank(productPublishedConf.getProductId())) {
                productPublishedConf.setProductId(productId);
            }
        }


        // ---------------------------------  反序列化 List (可在此加入业务逻辑) ---------------------------------
        HashMap<String, Object> frontColors = convertToFaceColors(uncheckedCast(newData.get("frontColors")), "front", productId);
        HashMap<String, Object> backColors = convertToFaceColors(uncheckedCast(newData.get("backColors")), "back", productId);
        HashMap<String, Object> lossed = convertToLossed(uncheckedCast(newData.get("productLoss")), productId);

        List<Map<String, Object>> maps = Stream.of(lossed, frontColors, backColors).filter(Objects::nonNull).collect(Collectors.toList());
        Map<String, Object> mergedMap = maps.stream()
                .flatMap(map -> map.entrySet().stream())
                .filter(entry -> entry.getValue() != null)
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (listA, listB) -> {
                    ArrayList<Map<String, Object>> mergedList = new ArrayList<>(uncheckedCast(listA));
                    mergedList.addAll(uncheckedCast(listB));
                    return mergedList;
                }));

        Set<Object> entitys = new HashSet<>();
        entitys.add(mergedMap);
        //entitys.add(product);
        entitys.add(productDetail);
        //entitys.add(productFace);
        entitys.add(productLooked);
        //entitys.add(productTechnology);
        entitys.add(productPublishedConf);
        persist.persistEntitys(entitys);

        TypedQuery<Product> query = entityManager.createQuery("SELECT s FROM Product s", Product.class);
        List<Product> products = query.getResultList();

        entityManager.flush();

//        String str = null;
//        str.length();

        HashMap<String, Object> map = new HashMap<>(5);
        map.put("save", "ok");
        map.put("productId", productId);

        return map;
    }

    @Override
    public HashMap<String, Object> queryProductList() {
        String sql = "SELECT\n" +
                "\tt.product_id,\n" +
                "\tt.product_category_id,\n" +
                "\tt.product_unit,\n" +
                "\tt.is_only_code,\n" +
                "\tt.brand_name,\n" +
                "\tpc.dispatch_multiple,\n" +
                "\tpc.row_num,\n" +
                "\tpt.product_technology_name\n" +
                "FROM\n" +
                "\tproduct t\n" +
                "\tLEFT JOIN product_published_conf pc ON t.product_id = pc.product_id\n" +
                "\tLEFT JOIN product_technology pt ON t.product_id = pt.product_id";
        ArrayList<Map<String, Object>> list = (ArrayList<Map<String, Object>>) this.sqlExecutorDao.queryToListMap(sql);
        HashMap<String, Object> map = new HashMap<>(3);
        map.put("productList", list);
        return map;
    }

    @Override
    public Product saveProduct(Product product) {
        product.setProductStatus("1");
        return productRepository.save(product);
    }

    @Override
    public ProductAllDataVo findProductAllData(String productId) {
        ProductAllDataVo productAllDataVo = new ProductAllDataVo();
        Product product = productRepository.findOne(productId);
        BeanUtil.copyProperties(product, productAllDataVo);

        ProductDetail productDetail = detailRepository.findByProductId(productId);
        BeanUtil.copyProperties(productDetail, productAllDataVo, "productId");

        ProductTechnology productTechnology = technologyRepository.findByProductId(productId);
        BeanUtil.copyProperties(productTechnology, productAllDataVo, "productId");

        ProductTypesetConf productTypesetConf = typesetConfRepository.findByProductId(productId);
        BeanUtil.copyProperties(productTypesetConf, productAllDataVo, "productId");

        ProductLooked productLooked = lookedRepository.findByProductId(productId);
        BeanUtil.copyProperties(productLooked, productAllDataVo, "productId");

        ProductPublishedConf publishedConf = publishedConfRepository.findByProductId(productId);
        BeanUtil.copyProperties(publishedConf, productAllDataVo, "productId");

        return productAllDataVo;
    }

    @Override
    public Map<String, Object> saveProductAllData(Product product, ProductDetail productDetail, ProductFace productFace, ProductLooked productLooked, ProductPublishedConf productPublishedConf, ProductTechnology productTechnology, List<ProductLoss> lossList, List<ProductProcess> processList, List<ProductFaceColor> frontFaceColorList, List<ProductFaceColor> backFaceColorList, List<ProductMaterial> productMaterialList) {
        product = this.saveProduct(product);
        productDetailApplication.saveProductDetail(product.getProductId(), productDetail);
        productFaceApplication.saveProductFace(product.getProductId(), productFace);
        productLookedApplication.saveProductLooked(product.getProductId(), productLooked);
        productPublishedConfApplication.saveProductPublishedConf(product.getProductId(), productPublishedConf);
        productTechnologyApplication.saveProductTechnology(product.getProductId(), productTechnology);
        productLossApplication.saveProductLossList(product.getProductId(), lossList);
        productProcessApplication.saveProductProcessList(productTechnology.getProductTechnologyId(), processList);
        productFaceColorApplication.saveProductFaceColorList(product.getProductId(), frontFaceColorList);
        productFaceColorApplication.saveProductFaceColorList(product.getProductId(), backFaceColorList);
        productMaterialApplication.saveProductMaterial(product.getProductId(), productMaterialList);
        Map<String, Object> map = new HashMap<>();
        map.put("productId", product.getProductId());
        map.put("productTechnologyId", productTechnology.getProductTechnologyId());
        return map;
    }

    @Override
    public Map<String, Object> slicedProductList(ProductQueryRequest queryRequest) {
        QueryDescriptor queryDescriptor = this.sqlExecutorDao.getQuery(QUERY_XML_FILE_PATH, "productList");
        QueryModel model = this.sqlExecutorDao.getQueryModel(queryDescriptor, queryRequest);
        return this.sqlExecutorDao.executeSlicedQuery(model);
    }

    @Override
    public void updateAProductStatus(List<String> ids, String status) {
        ids.forEach(id -> {
            Product product = productRepository.findOne(id);
            product.setProductStatus(status);
            productRepository.save(product);
        });
    }

    /**
     * 转为正式产品
     */
    @Override
    public void ConversionToOfficialProducts(String ids) {
        List<String> list;
        try {
             list = objectMapper.readValue(ids, new TypeReference<List<String>>() {
            });
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }

        List<ProductReturn> productReturnList = new ArrayList<>();
        for (String productId : list) {
            Product product = productRepository.findOne(productId);
            product.setProductType("正式产品");
            productRepository.save(product);

            // TODO 发送到SAP

            // 查询BOM信息
            List<ProductMaterial> productMaterialList = materialRepository.findByProductId(productId);
            // 查询基本信息 - 已查
            // 查询商品外观 - 获取单层厚、单重
            ProductLooked productLooked = lookedRepository.findByProductId(productId);
            // 查询排版参数配置 - 获取排版行数、派工倍数
            ProductPublishedConf publishedConf = publishedConfRepository.findByProductId(productId);
            List<Map<String, Object>> returnBOM = null;
            try {
                returnBOM = getReturnBOM(productMaterialList);
            } catch (JsonProcessingException e) {
                throw new RuntimeException(e);
            }
            ProductReturn productReturn = new ProductReturn();
            BeanUtil.copyProperties(product, productReturn);
            BeanUtil.copyProperties(productLooked, productReturn);
            BeanUtil.copyProperties(publishedConf, productReturn);
            productReturn.setBOMItems(returnBOM);

            // 封装数据
            productReturnList.add(productReturn);
        }
        // 发送给SAP
        // TODO 接口未定、返回数据未解析
        String url = SystemCache.getParameter("sap.service.null.url", String.class);
        try {
            String execute = httpClient.execute(productReturnList, url);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private List<ProductReturn> getReturnList(Product product, List<ProductMaterial> productMaterialList, ProductLooked productLooked, ProductPublishedConf publishedConf) {

        return null;
    }

    @SuppressWarnings("unchecked")
    public static <T> T uncheckedCast(Object obj) {
        if (obj != null) {
            return (T) obj;
        }
        return null;
    }

    public HashMap<String, Object> convertToFaceColors(Map<String, ArrayList<Map<String, Object>>> map, String type, String productId) {
        if (map == null) {
            return null;
        }
        HashMap<String, Object> retult = new HashMap<>(5);
        retult.put("add", convertColor(map.get("add"), type, productId));
        retult.put("up", convertColor(map.get("up"), type, productId));
        retult.put("del", convertColor(map.get("del"), type, productId));
        return retult;
    }

    public ArrayList<ProductFaceColor> convertColor(ArrayList<Map<String, Object>> list, String type, String productId) {
        if (list.size() == 0) {
            return null;
        }
        list.forEach(faceColor -> {
            faceColor.put("productId", productId);
            faceColor.put("face", type);
        });
        return objectMapper.convertValue(list, new TypeReference<ArrayList<ProductFaceColor>>() {
        });
    }

    private HashMap<String, Object> convertToProcesseds(Map<String, ArrayList<Map<String, Object>>> map, String productTechnologyId) {
        if (map == null) {
            return null;
        }
        HashMap<String, Object> retult = new HashMap<>(5);
        retult.put("add", convertProcessed(map.get("add"), productTechnologyId));
        retult.put("up", convertProcessed(map.get("up"), productTechnologyId));
        retult.put("del", convertProcessed(map.get("del"), productTechnologyId));
        return retult;
    }

    public ArrayList<ProductProcess> convertProcessed(ArrayList<Map<String, Object>> list, String productTechnologyId) {
        if (list.size() == 0) {
            return null;
        }
        list.forEach(process -> {
            process.put("productTechnologyId", productTechnologyId);
        });
        return objectMapper.convertValue(list, new TypeReference<ArrayList<ProductProcess>>() {
        });
    }

    private HashMap<String, Object> convertToLossed(Map<String, ArrayList<Map<String, Object>>> map, String productId) {
        if (map == null) {
            return null;
        }
        HashMap<String, Object> retult = new HashMap<>(5);
        retult.put("add", convertLossed(map.get("add"), productId));
        retult.put("up", convertLossed(map.get("up"), productId));
        retult.put("del", convertLossed(map.get("del"), productId));
        return retult;
    }

    public ArrayList<ProductLoss> convertLossed(ArrayList<Map<String, Object>> list, String productId) {
        if (list.size() == 0) {
            return null;
        }
        if (StringUtil.isNotBlank(productId)) {
            list.forEach(loss -> {
                loss.put("productId", productId);
            });
        }
        return objectMapper.convertValue(list, new TypeReference<ArrayList<ProductLoss>>() {
        });
    }

    public List<Map<String, Object>> getReturnBOM(List<ProductMaterial> productMaterialList) throws JsonProcessingException {
        List<Map<String, Object>> returnBOMList = new ArrayList<>();
        for (ProductMaterial productMaterial : productMaterialList) {
            Map<String, Object> map = new HashMap<>();
            map.put("IDNRK", productMaterial.getProductMaterialId());
            map.put("MENGE", productMaterial.getMaterialNumber());
            map.put("MEINS", productMaterial.getMaterialUnit());
            map.put("AUSCH", productMaterial.getMaterialAusch());
            map.put("POTX1", productMaterial.getMaterialPotx1());
            map.put("POTX2", productMaterial.getMaterialJson());
            map.put("ALPGR", productMaterial.getMaterialAlpgr());
            map.put("ALPRF", productMaterial.getMaterialAlprf());
            map.put("ALPST", productMaterial.getMaterialAlpst());
            map.put("EWAHR", productMaterial.getMaterialEwahr());
            returnBOMList.add(map);
        }
        return returnBOMList;
    }


}

//        resultMap.put("selectedBrand", convertToMap(Brand.class, "brandName", "brandId", brands));
//        resultMap.put("selectedFactory", convertToMap(Factory.class, "factoryName", "factoryId", factorys));
//        resultMap.put("selectedColor", convertToMap(Color.class, "colorName", "colorId", colors));
//        resultMap.put("selectedCategory", convertToMap(ProductCategory.class, "productCategoryId",
//                "productCategoryName", categorys));


//    public ArrayList<Map<String, Object>> findProductPage(int page, int size) {
////        PageRequest pageRequest = new PageRequest(page, size);
////        Page<Product> productPage = productRepository.findAll(pageRequest);
////        mapList.getContent().stream().map(Product::toString).forEach(System.out::println);
//        ArrayList<Product> productPage = (ArrayList<Product>) productRepository.findAll();
//
////        convert = mapList.stream().collect(Collectors.toMap(index -> index, Function.identity(),
////                (oldV, newV) -> oldV));
//        return objectMapper.convertValue(productPage, new TypeReference<ArrayList<Map<String, Object>>>() {});
//    }
//


//        if(product != null) {
//            BigDecimal id = product.getProductId();
//            if(id == null) {
//                product.setProductId(String.valueOf(Snowflake.nextId()));
//                entityManager.persist(product);
//            }
//            entityManager.merge(product);
//        }
//        if(productDetail != null) {
//            BigDecimal id = productDetail.getProductDetailId();
//            if(id == null) {
//                productDetail.setProductDetailId(String.valueOf(Snowflake.nextId()));
//                entityManager.persist(productDetail);
//            }
//            entityManager.merge(productDetail);
//        }
//        if(productLooked != null) {
//            BigDecimal id = productLooked.getProductLookedId();
//            if(id == null) {
//                productLooked.setProductLookedId(String.valueOf(Snowflake.nextId()));
//                entityManager.persist(productLooked);
//            }
//            entityManager.merge(productLooked);
//        }
