package com.ximai.mes.cal.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ximai.common.annotation.Log;
import com.ximai.common.constant.UserConstants;
import com.ximai.common.core.controller.BaseController;
import com.ximai.common.core.domain.AjaxResult;
import com.ximai.common.core.page.TableDataInfo;
import com.ximai.common.enums.BusinessType;
import com.ximai.common.utils.data.StringUtils;
import com.ximai.mes.cal.domain.CalPlan;
import com.ximai.mes.cal.domain.CalPlanWorkunit;
import com.ximai.mes.cal.domain.CalShift;
import com.ximai.mes.cal.domain.vo.CalShiftVo;
import com.ximai.mes.cal.mapper.CalShiftMapper;
import com.ximai.mes.cal.service.ICalPlanService;
import com.ximai.mes.cal.service.ICalShiftService;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * 计划班次Controller
 *
 * @date 2022-06-06
 */
@RestController
@RequestMapping("/mes/cal/shift")
public class CalShiftController extends BaseController {
    @Autowired
    private ICalShiftService calShiftService;

    @Resource
    private CalShiftMapper calShiftMapper;


    @Autowired
    private ICalPlanService calPlanService;


    /**
     * 查询计划班次列表
     */
    @GetMapping("/list")
    public TableDataInfo list(CalShift calShift) {
        startPage();
        List<CalShift> list = calShiftService.selectCalShiftList(calShift);
        return getDataTable(list);
    }


    /**
     * 获取计划班次详细信息
     */
    @PreAuthorize("@ss.hasPermi('mes:cal:calplan:query')")
    @GetMapping(value = "/{shiftId}")
    public AjaxResult getInfo(@PathVariable("shiftId") Long shiftId) {
        return AjaxResult.success(calShiftService.selectCalShiftByShiftId(shiftId));
    }


    @ApiOperation("默认班次查询")
    @Log(title = "默认班次查询", businessType = BusinessType.QUERY)
    @GetMapping(value = "getDetailShift/{shiftType}")
    public AjaxResult getDetailShift(@PathVariable("shiftType") String shiftType) {
        return AjaxResult.success(calShiftService.getDetailShift(shiftType));
    }


    @ApiOperation("查询上工班次")
    @Log(title = "查询上工班次", businessType = BusinessType.QUERY)
    @GetMapping(value = "/listOnWorkShift")
    public TableDataInfo listOnWorkShift(CalPlanWorkunit calPlanWorkunit) {
        startPage();
        QueryWrapper<CalShiftVo> calShiftVoQueryWrapper = new QueryWrapper<>();
        calShiftVoQueryWrapper.eq(StringUtils.isNotEmpty(calPlanWorkunit.getShiftName()), "ash.shift_name", calPlanWorkunit.getShiftName());
        calShiftVoQueryWrapper.eq(StringUtils.isNotEmpty(calPlanWorkunit.getWorkunitId()), "cpw.workunit_id", calPlanWorkunit.getWorkunitId());
        calShiftVoQueryWrapper.eq(StringUtils.isNotEmpty(calPlanWorkunit.getTheDay()), "cpw.the_day", calPlanWorkunit.getTheDay());
        calShiftVoQueryWrapper.eq(StringUtils.isNotEmpty(calPlanWorkunit.getShiftName()), "ash.shift_name", calPlanWorkunit.getShiftName());


        calShiftVoQueryWrapper.like(StringUtils.isNotEmpty(calPlanWorkunit.getWorkstationName()), "mws.workstation_name", calPlanWorkunit.getWorkstationName());
        calShiftVoQueryWrapper.like(StringUtils.isNotEmpty(calPlanWorkunit.getWorkshopName()), "mws.workshop_name", calPlanWorkunit.getWorkshopName());
        return getDataTable(calShiftMapper.selectOnWorkShiftByQw(calShiftVoQueryWrapper));
    }


    /**
     * 新增计划班次
     */
    @PreAuthorize("@ss.hasPermi('mes:cal:calplan:add')")
    @Log(title = "计划班次", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody CalShift calShift) {
        int count = calShiftService.checkShiftCount(calShift.getPlanId());
        CalPlan plan = calPlanService.selectCalPlanByPlanId(calShift.getPlanId());
        if (UserConstants.CAL_SHIFT_TYPE_SINGLE.equals(plan.getShiftType()) && count > 0) {
            return AjaxResult.error("轮班方式为 白班 时只能有一个班次！");
        }
        if (UserConstants.CAL_SHIFT_TYPE_TWO.equals(plan.getShiftType()) && count > 1) {
            return AjaxResult.error("轮班方式为 两班倒 时只能有两个班次！");
        }
        if (UserConstants.CAL_SHIFT_TYPE_THREE.equals(plan.getShiftType()) && count > 2) {
            return AjaxResult.error("轮班方式为 三班倒 时只能有三个班次！");
        }

        return toAjax(calShiftService.insertCalShift(calShift));
    }

    /**
     * 修改计划班次
     */
    @PreAuthorize("@ss.hasPermi('mes:cal:calplan:edit')")
    @Log(title = "计划班次修改", businessType = BusinessType.UPDATE)
    @ApiOperation("计划班次修改")
    @PutMapping
    public AjaxResult edit(@RequestBody CalShift calShift) {
        return toAjax(calShiftService.updateCalShift(calShift));
    }

    /**
     * 删除计划班次
     */
    @ApiOperation("计划班次删除")
    @PreAuthorize("@ss.hasPermi('mes:cal:calplan:remove')")
    @Log(title = "计划班次删除", businessType = BusinessType.DELETE)
    @DeleteMapping("/{shiftIds}")
    public AjaxResult remove(@PathVariable Long[] shiftIds) {
        return toAjax(calShiftService.deleteCalShiftByShiftIds(shiftIds));
    }

}
