package com.ximai.common.core.controller;

import com.ximai.common.constant.HttpStatus;
import com.ximai.common.core.domain.AjaxResult;
import com.ximai.common.core.domain.PageParams;
import com.ximai.common.core.domain.SapAjaxResult;
import com.ximai.common.core.domain.SapResult;
import com.ximai.common.core.domain.model.LoginUser;
import com.ximai.common.core.page.PageDomain;
import com.ximai.common.core.page.PaginationHelper;
import com.ximai.common.core.page.TableDataInfo;
import com.ximai.common.core.page.TableSupport;
import com.ximai.common.utils.PageUtils;
import com.ximai.common.utils.SecurityUtils;
import com.ximai.common.utils.data.DateUtils;
import com.ximai.common.utils.data.StringUtils;
import com.ximai.common.utils.sql.SqlUtil;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;

import java.beans.PropertyEditorSupport;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * web层通用数据处理
 */
public class BaseController {
    protected final Logger logger = LoggerFactory.getLogger(this.getClass());

    /**
     * 将前台传递过来的日期格式的字符串，自动转化为Date类型
     */
    @InitBinder
    public static void initBinder(final WebDataBinder binder) {
        // Date 类型转换
        binder.registerCustomEditor(Date.class, new PropertyEditorSupport() {
            @Override
            public void setAsText(final String text) {
                setValue(DateUtils.parseDate(text));
            }
        });
    }

    /**
     * 设置请求分页数据
     */
    protected static void startPage() {
        PageUtils.startPage();
    }

    protected static void startPage(final PageParams pageParams) {
        PageUtils.startPage(pageParams);
    }

    /**
     * 设置请求排序数据
     */
    protected static void startOrderBy() {
        final PageDomain pageDomain = TableSupport.buildPageRequest();
        if (StringUtils.isNotEmpty(pageDomain.getOrderBy())) {
            final String orderBy = SqlUtil.escapeOrderBySql(pageDomain.getOrderBy());
            PageHelper.orderBy(orderBy);
        }
    }

    /**
     * 清理分页的线程变量
     */
    protected static void clearPage() {
        PageUtils.clearPage();
    }

    /**
     * 响应请求分页数据
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    protected static <T> TableDataInfo<T> getDataTable(final List<T> list) {
        final TableDataInfo<T> rspData = new TableDataInfo<>(list, new PageInfo<>(list).getTotal());
        rspData.setCode(HttpStatus.SUCCESS);
        rspData.setMsg("查询成功");
        return rspData;
    }

    /**
     * 返回成功
     */
    public static AjaxResult success() {
        return AjaxResult.success();
    }

    /**
     * 返回失败消息
     */
    public AjaxResult error() {
        return AjaxResult.error();
    }

    /**
     * 返回成功消息
     */
    public static AjaxResult success(final String message) {
        return AjaxResult.success(message);
    }

    /**
     * 返回失败消息
     */
    public AjaxResult error(final String message) {
        return AjaxResult.error(message);
    }

    /**
     * 响应返回结果
     *
     * @param rows 影响行数
     * @return 操作结果
     */
    protected static AjaxResult toAjax(final int rows) {
        return rows > 0 ? AjaxResult.success() : AjaxResult.error();
    }

    /**
     * 响应返回结果
     *
     * @param rows 影响行数
     * @return 操作结果
     */
    protected static SapAjaxResult sapToAjax(final int rows) {
        return rows > 0 ? SapAjaxResult.success() : SapAjaxResult.error();
    }

    /**
     * 响应返回结果
     *
     * @param result 结果
     * @return 操作结果
     */
    protected AjaxResult toAjax(final boolean result) {
        return result ? success() : error();
    }

    /**
     * @param sapResult
     * @return 操作结果
     */
    protected AjaxResult toAjax(final SapResult sapResult) {
        return "S".equals(sapResult.getType()) ? success(sapResult.getMessage()) : error(sapResult.getMessage());
    }

    /**
     * 页面跳转
     */
    public String redirect(final String url) {
        return StringUtils.format("redirect:{}", url);
    }

    /**
     * 获取用户缓存信息
     */
    public LoginUser getLoginUser() {
        return SecurityUtils.getLoginUser();
    }

    /**
     * 获取登录用户id
     */
    public Long getUserId() {
        return getLoginUser().getUserId();
    }

    /**
     * 获取登录部门id
     */
    public Long getDeptId() {
        return getLoginUser().getDeptId();
    }

    /**
     * 获取登录用户名
     */
    public String getUsername() {
        return getLoginUser().getUsername();
    }
}
