package com.ximai.mes.md.controller;

import cn.hutool.core.bean.BeanUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ximai.common.annotation.Log;
import com.ximai.common.core.controller.BaseController;
import com.ximai.common.core.domain.AjaxResult;
import com.ximai.common.core.page.TableDataInfo;
import com.ximai.common.enums.BusinessType;
import com.ximai.common.utils.SecurityUtils;
import com.ximai.common.utils.excel.ExcelWriter;
import com.ximai.mes.md.domain.MdWorkunit;
import com.ximai.mes.md.dto.MdWorkunitExcelExport;
import com.ximai.mes.md.mapper.MdWorkunitMapper;
import com.ximai.mes.md.service.IMdWorkunitService;
import com.ximai.mes.md.vo.MdItemVo;
import com.ximai.mes.md.vo.MdWorkunitVo;
import com.ximai.mes.tm.service.ITmToolMachinesService;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;

/**
 * 工作单元Controller
 *
 * @author yinjinlu
 * @date 2024-01-18
 */
@RestController
@RequestMapping("/md/workunit")
public class MdWorkunitController extends BaseController {
    @Autowired
    private IMdWorkunitService mdWorkunitService;


    @Autowired
    private MdWorkunitMapper mdWorkunitMapper;

    @Autowired
    private ITmToolMachinesService tmToolMachinesService;

    /**
     * 查询工作单元列表
     */
    @ApiOperation(value = "分页查询工作单元")
    @PreAuthorize("@ss.hasPermi('md:workunit:list')")
    @GetMapping("/list")
    public TableDataInfo list(MdWorkunit mdWorkunit) {
        startPage();
        List<MdWorkunitVo> list = mdWorkunitService.selectMdWorkunitList(mdWorkunit);
        return getDataTable(list);
    }


    /**
     * 弹窗
     */
    @ApiOperation(value = "分页查询工作单元")
    @PreAuthorize("@ss.hasPermi('md:workunit:list')")
    @GetMapping("/lists")
    public TableDataInfo lists(MdWorkunit mdWorkunit) {
        startPage();
        List<MdWorkunitVo> list = mdWorkunitMapper.selectMdWorkunitLists(mdWorkunit);
        return getDataTable(list);
    }

    /**
     * 获取工作单元详细信息
     */
    @PreAuthorize("@ss.hasPermi('md:workunit:query')")
    @GetMapping(value = "/{workunitId}")
    public AjaxResult getInfo(@PathVariable("workunitId") Long workunitId) {
        return AjaxResult.success(mdWorkunitService.selectMdWorkunitByWorkunitId(workunitId));
    }

    /**
     * 新增工作单元
     */
    @PreAuthorize("@ss.hasPermi('md:workunit:add')")
    @Log(title = "工作单元", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody MdWorkunit mdWorkunit) {
        return toAjax(mdWorkunitService.insertMdWorkunit(mdWorkunit));
    }

    /**
     * 修改工作单元
     */
    @PreAuthorize("@ss.hasPermi('md:workunit:edit')")
    @Log(title = "工作单元", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody MdWorkunit mdWorkunit) {
        return toAjax(mdWorkunitService.updateMdWorkunit(mdWorkunit));
    }

    /**
     * 删除工作单元
     */
    @PreAuthorize("@ss.hasPermi('md:workunit:remove')")
    @Log(title = "工作单元", businessType = BusinessType.DELETE)
    @DeleteMapping("/{workunitIds}")
    public AjaxResult remove(@PathVariable Long[] workunitIds) {
        return toAjax(mdWorkunitService.deleteMdWorkunitByWorkunitIds(workunitIds));
    }

    @ApiOperation(value = "根据当前登录人员查询工作单元")
    @GetMapping("/getListByUser")
    public TableDataInfo getListByUser() {

        List<MdWorkunitVo> list = mdWorkunitService.selectMdWorkunitListByUser(new QueryWrapper<MdWorkunit>().eq("user_name", SecurityUtils.getUsername()));
        return getDataTable(list);
    }


    @ApiOperation(value = "根据工作单元查询上机的刀模版信息")
    @GetMapping("/getUpTmToolRequestList")
    public TableDataInfo getUpTmToolRequestList(MdWorkunitVo vo) {
        startPage();
        List<MdItemVo> list = tmToolMachinesService.selectUpTmToolRequestList(vo);
        return getDataTable(list);
    }
    /**
     * 导出工作单元列表
     */
    @PreAuthorize("@ss.hasPermi('md:workunit:export')")
    @Log(title = "工作单元", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, MdWorkunit mdWorkunit) throws IOException {
        List<MdWorkunitVo> list = mdWorkunitService.selectMdWorkunitList(mdWorkunit);
        List<MdWorkunitExcelExport> listExport = BeanUtil.copyToList(list, MdWorkunitExcelExport.class);
        ExcelWriter.write(response, MdWorkunitExcelExport.class, listExport);
    }
}
