package com.ximai.mes.wm.controller.mobile;

import com.ximai.common.annotation.Log;
import com.ximai.common.constant.UserConstants;
import com.ximai.common.core.controller.BaseController;
import com.ximai.common.core.domain.AjaxResult;
import com.ximai.common.core.page.TableDataInfo;
import com.ximai.common.enums.BusinessType;
import com.ximai.common.utils.data.StringUtils;
import com.ximai.mes.md.domain.MdVendor;
import com.ximai.mes.md.service.IMdVendorService;
import com.ximai.mes.wm.domain.WmRtVendor;
import com.ximai.mes.wm.domain.tx.RtVendorTxBean;
import com.ximai.mes.wm.service.IStorageCoreService;
import com.ximai.mes.wm.service.IWmRtVendorService;
import com.ximai.system.strategy.AutoCodeUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@Api("退回供应商")
@RestController
@RequestMapping("/mobile/wm/rtvendor")
public class WmRtVendorMobController extends BaseController {

    @Autowired
    private IWmRtVendorService wmRtVendorService;

    @Autowired
    private IStorageCoreService storageCoreService;

    @Autowired
    private IMdVendorService mdVendorService;

    @Autowired
    private AutoCodeUtil autoCodeUtil;

    /**
     * 查询供应商退货列表
     */
    @ApiOperation("查询采购退货单据清单")
    @PreAuthorize("@ss.hasPermi('mes:wm:rtvendor:list')")
    @GetMapping("/list")
    public TableDataInfo list(WmRtVendor wmRtVendor)
    {
        startPage();
        List<WmRtVendor> list = wmRtVendorService.selectWmRtVendorList(wmRtVendor);
        return getDataTable(list);
    }

    /**
     * 获取供应商退货详细信息
     */
    @ApiOperation("查询采购退货单据明细")
    @PreAuthorize("@ss.hasPermi('mes:wm:rtvendor:query')")
    @GetMapping(value = "/{rtId}")
    public AjaxResult getInfo(@PathVariable("rtId") Long rtId)
    {
        return AjaxResult.success(wmRtVendorService.selectWmRtVendorByRtId(rtId));
    }

    /**
     * 新增供应商退货
     */
    @ApiOperation("新增采购退货单")
    @PreAuthorize("@ss.hasPermi('mes:wm:rtvendor:add')")
    @Log(title = "供应商退货", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody WmRtVendor wmRtVendor)
    {
        if(StringUtils.isNotNull(wmRtVendor.getRtCode())){
            if(UserConstants.NOT_UNIQUE.equals(wmRtVendorService.checkCodeUnique(wmRtVendor))){
                return AjaxResult.error("退货单编号已经存在！");
            }
        }else {
            wmRtVendor.setRtCode(autoCodeUtil.genSerialCode(UserConstants.WM_RTVENDOR_CODE,""));
        }

        if(StringUtils.isNotNull(wmRtVendor.getVendorId())){
            MdVendor vendor = mdVendorService.selectMdVendorByVendorId(wmRtVendor.getVendorId());
            wmRtVendor.setVendorCode(vendor.getVendorCode());
            wmRtVendor.setVendorName(vendor.getVendorName());
            wmRtVendor.setVendorNick(vendor.getVendorNick());
        }

        wmRtVendor.setCreateBy(getUsername());
        wmRtVendorService.insertWmRtVendor(wmRtVendor);
        return AjaxResult.success(wmRtVendor);
    }

    /**
     * 修改供应商退货
     */
    @ApiOperation("编辑采购退货单")
    @PreAuthorize("@ss.hasPermi('mes:wm:rtvendor:edit')")
    @Log(title = "供应商退货", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody WmRtVendor wmRtVendor)
    {
        if(UserConstants.NOT_UNIQUE.equals(wmRtVendorService.checkCodeUnique(wmRtVendor))){
            return AjaxResult.error("退货单编号已经存在！");
        }

        if(StringUtils.isNotNull(wmRtVendor.getVendorId())){
            MdVendor vendor = mdVendorService.selectMdVendorByVendorId(wmRtVendor.getVendorId());
            wmRtVendor.setVendorCode(vendor.getVendorCode());
            wmRtVendor.setVendorName(vendor.getVendorName());
            wmRtVendor.setVendorNick(vendor.getVendorNick());
        }

        return toAjax(wmRtVendorService.updateWmRtVendor(wmRtVendor));
    }


    /**
     * 执行退货
     */
    @ApiOperation("执行退货")
    @PreAuthorize("@ss.hasPermi('mes:wm:rtvendor:edit')")
    @Log(title = "供应商退货单", businessType = BusinessType.UPDATE)
    @Transactional
    @PutMapping("/{rtId}")
    public AjaxResult execute(@PathVariable Long rtId){
        //判断单据状态
        WmRtVendor wmRtVendor = wmRtVendorService.selectWmRtVendorByRtId(rtId);

        //构造事务Bean
        List<RtVendorTxBean> beans = wmRtVendorService.getTxBeans(rtId);

        //调用库存核心
        storageCoreService.processRtVendor(beans);

        //更新单据状态
        WmRtVendor rtVendor = wmRtVendorService.selectWmRtVendorByRtId(rtId);
        rtVendor.setStatus(UserConstants.ORDER_STATUS_FINISHED);
        wmRtVendorService.updateWmRtVendor(rtVendor);

        return AjaxResult.success();
    }
}
