package com.ximai.mes.cal.controller;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ximai.common.annotation.Log;
import com.ximai.common.constant.UserConstants;
import com.ximai.common.core.controller.BaseController;
import com.ximai.common.core.domain.AjaxResult;
import com.ximai.common.core.page.TableDataInfo;
import com.ximai.common.enums.BusinessType;
import com.ximai.common.utils.MessageUtils;
import com.ximai.common.utils.SecurityUtils;
import com.ximai.common.utils.data.DateUtils;
import com.ximai.common.utils.data.ExceptionUtil;
import com.ximai.common.utils.excel.ExcelWriter;
import com.ximai.mes.cal.domain.CalPlan;
import com.ximai.mes.cal.domain.CalShift;
import com.ximai.mes.cal.dto.CalPlanExcelExport;
import com.ximai.mes.cal.mapper.CalPlanMapper;
import com.ximai.mes.cal.service.ICalPlanService;
import com.ximai.mes.cal.service.ICalPlanTeamService;
import com.ximai.mes.cal.service.ICalShiftService;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

import static com.ximai.common.utils.data.DateUtils.formatShiftDate;

/**
 * 排班计划Controller
 *
 * @date 2022-06-06
 */
@RestController
@RequestMapping("/mes/cal/calplan")
public class CalPlanController extends BaseController {
    @Autowired
    private ICalPlanService calPlanService;

    @Resource
    private CalPlanMapper calPlanMapper;
    @Autowired
    private ICalShiftService calShiftService;

    @Autowired
    private ICalPlanTeamService calPlanTeamService;

    /**
     * 查询排班计划列表
     */
    @PreAuthorize("@ss.hasPermi('mes:cal:calplan:list')")
    @GetMapping("/list")
    public TableDataInfo list(CalPlan calPlan) {
        startPage();
        List<CalPlan> list = calPlanService.selectCalPlanList(calPlan);
        return getDataTable(list);
    }

    /**
     * 获取排班计划详细信息
     */
    @PreAuthorize("@ss.hasPermi('mes:cal:calplan:query')")
    @GetMapping(value = "/{planId}")
    public AjaxResult getInfo(@PathVariable("planId") Long planId) {
        return AjaxResult.success(calPlanService.selectCalPlanByPlanId(planId));
    }

    /**
     * 新增排班计划
     */
    @PreAuthorize("@ss.hasPermi('mes:cal:calplan:add')")
    @Log(title = "排班计划新增", businessType = BusinessType.INSERT)
    @Transactional
    @PostMapping
    @ApiOperation("排班计划新增")
    public AjaxResult add(@RequestBody CalPlan calPlan) {
        List<CalPlan> calPlans = calPlanMapper.selectListByQw(new QueryWrapper<CalPlan>().eq("plan_code", calPlan.getPlanCode()));
        ExceptionUtil.checkTrueThrowException(CollectionUtil.isNotEmpty(calPlans), MessageUtils.message("cal.error.error1"));

        checkPlan(calPlan);

        // 插入方案
        calPlanService.insertCalPlan(calPlan);
        calPlanService.updateRelationship(calPlan);
        calPlanService.insertCalWorkunits(calPlan);
        return AjaxResult.success();
    }


    /**
     * 修改排班
     */
    @ApiOperation("排班计划修改完成")
    @PreAuthorize("@ss.hasPermi('mes:cal:calplan:edit')")
    @Log(title = "排班计划修改", businessType = BusinessType.UPDATE)
    @Transactional
    @PutMapping
    public AjaxResult edit(@RequestBody CalPlan calPlan) {
        checkPlan(calPlan);

        calPlanService.updateCalPlan(calPlan);
        calPlanService.updateRelationship(calPlan);
        calPlanService.insertCalWorkunits(calPlan);
        return AjaxResult.success();
    }

    private static void checkPlan(CalPlan calPlan) {
        List<CalShift> calShifts = calPlan.getCalShifts();
        ExceptionUtil.checkTrueThrowException(calShifts == null || CollectionUtil.isEmpty(calShifts), MessageUtils.message("cal.error.error2"));
        Date planStartDate = calPlan.getStartDate();
        Date planEndDate = calPlan.getEndDate();
        ExceptionUtil.checkTrueThrowException(planStartDate == null || planEndDate == null, MessageUtils.message("cal.error.error3"));
        ExceptionUtil.checkTrueThrowException(planStartDate.getTime() > planEndDate.getTime(), MessageUtils.message("cal.error.error4"));
        ExceptionUtil.checkTrueThrowException(CollectionUtil.isEmpty(calShifts), MessageUtils.message("cal.error.error5"));


        long count = calShifts.stream().filter(x ->
                (x.getEndTime() == null && x.getEndTime() == "") ||
                        (x.getStartTime() == null && x.getStartTime() == "")
        ).count();
        ExceptionUtil.checkTrueThrowException(count > 0, MessageUtils.message("cal.error.error5"));


        String shiftType = calPlan.getShiftType();
        int size = calShifts.size();
        ExceptionUtil.checkTrueThrowException(UserConstants.CAL_SHIFT_TYPE_SINGLE.equals(shiftType) && size != 1, MessageUtils.message("cal.error.error7"));

        ExceptionUtil.checkTrueThrowException(UserConstants.CAL_SHIFT_TYPE_TWO.equals(shiftType) && size != 2, MessageUtils.message("cal.error.error7"));

        ExceptionUtil.checkTrueThrowException(UserConstants.CAL_SHIFT_TYPE_THREE.equals(shiftType) && size != 3, MessageUtils.message("cal.error.error8"));

        Date shiftStartDate = null;
        Date shiftEndDate = null;
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        String nowDateFormat = sdf.format(planStartDate);
        Set<Integer> collect2 = calShifts.stream().map(CalShift::getOrderNum).filter(Objects::nonNull).collect(Collectors.toSet());
        ExceptionUtil.checkTrueThrowException(collect2.size() != calShifts.size(), MessageUtils.message("cal.error.error9"));

        for (CalShift calShift : calShifts) {
            String shiftName = calShift.getShiftName();
            String nextDateFormat = null;
            String endTime = calShift.getEndTime();
            shiftStartDate = formatShiftDate(nowDateFormat, calShift.getStartTime());
            shiftEndDate = formatShiftDate(nowDateFormat, endTime);
            if (Objects.equals(shiftName, "夜班")) {
                if (shiftStartDate.getTime() > shiftEndDate.getTime()) {
                    Date nextDay = DateUtils.getNextDay(planStartDate);

                    //今天日期
                    sdf = new SimpleDateFormat("yyyy-MM-dd");
                    nextDateFormat = sdf.format(nextDay);
                    shiftEndDate = formatShiftDate(nextDateFormat, endTime);

                }
            }
            ExceptionUtil.checkTrueThrowException(planStartDate.getTime() >= shiftStartDate.getTime(), shiftName + MessageUtils.message("cal.error.error22"));
            ExceptionUtil.checkTrueThrowException(planEndDate.getTime() <= shiftEndDate.getTime(), shiftName + MessageUtils.message("cal.error.error23"));

            calShift.setExecEndTime(shiftEndDate);
            calShift.setExecStartTime(shiftStartDate);
            ExceptionUtil.checkTrueThrowException(shiftEndDate.getTime() < shiftStartDate.getTime(),shiftName+ MessageUtils.message("cal.error.error24", shiftName));

            List<CalShift> breakTimeList = calShift.getBreakTimeList();
            if (CollectionUtil.isNotEmpty(breakTimeList)) {
                Set<Integer> collect = breakTimeList.stream().map(CalShift::getOrderNum).filter(Objects::nonNull).collect(Collectors.toSet());
                ExceptionUtil.checkTrueThrowException(collect.size() != breakTimeList.size(), MessageUtils.message("cal.error.error10"));


                ExceptionUtil.checkTrueThrowException(shiftEndDate.getTime() <= shiftStartDate.getTime(), shiftName + MessageUtils.message("cal.error.error24", shiftName));


                breakTimeList.sort(Comparator.comparing(CalShift::getOrderNum));
                if (breakTimeList.size() == 2) {
                    CalShift calShift1 = breakTimeList.get(0);
                    CalShift calShift2 = breakTimeList.get(1);
                    String breakStartTime1 = calShift1.getBreakStartTime();
                    String breakEndTime1 = calShift1.getBreakEndTime();
                    ExceptionUtil.checkTrueThrowException(Objects.equals(breakStartTime1, breakEndTime1), MessageUtils.message("cal.error.error25"));
                    String breakStartTime2 = calShift2.getBreakStartTime();
                    String breakEndTime2 = calShift2.getBreakEndTime();
                    ExceptionUtil.checkTrueThrowException(Objects.equals(breakStartTime2, breakEndTime2), MessageUtils.message("cal.error.error25"));

                    Date breakStartDate1 = formatShiftDate(nowDateFormat, breakStartTime1);
                    Date breakEndDate1 = formatShiftDate(nowDateFormat, breakEndTime1);
                    Date breakStartDate2 = formatShiftDate(nowDateFormat, breakStartTime2);
                    Date breakEndDate2 = formatShiftDate(nowDateFormat, breakEndTime2);


                    Integer nextDay1 = calShift1.getNextDay();
                    if (nextDay1 != null && nextDay1 == 1) {
                        if (breakEndDate1.after(breakStartDate1)) {
                            breakStartDate1 = formatShiftDate(nextDateFormat, breakStartTime1);
                            breakEndDate1 = formatShiftDate(nextDateFormat, breakEndTime1);
                        } else {
                            breakEndDate1 = formatShiftDate(nextDateFormat, breakEndTime1);
                        }
                    }

                    Integer nextDay2 = calShift2.getNextDay();
                    if (nextDay2 != null && nextDay2 == 1) {
                        if (breakEndDate2.after(breakStartDate2)) {
                            breakStartDate2 = formatShiftDate(nextDateFormat, breakStartTime2);
                            breakEndDate2 = formatShiftDate(nextDateFormat, breakEndTime2);
                        } else {
                            breakEndDate2 = formatShiftDate(nextDateFormat, breakEndTime2);
                        }
                    }


                    ExceptionUtil.checkTrueThrowException(
                            !breakEndDate2.after(breakStartDate1),
                            shiftName + MessageUtils.message("cal.error.error28"));

                    ExceptionUtil.checkTrueThrowException(shiftStartDate.after(breakStartDate1), shiftName + MessageUtils.message("cal.error.error26"));
                    ExceptionUtil.checkTrueThrowException(breakStartDate2.after(shiftEndDate), shiftName + MessageUtils.message("cal.error.error27"));

                    boolean afterA = breakEndDate1.after(breakStartDate1);
                    boolean afterB = breakStartDate2.after(breakEndDate1);
                    boolean afterC = breakEndDate2.after(breakStartDate2);
                    ExceptionUtil.checkTrueThrowException(
                            !(afterA && afterB && afterC),
                             MessageUtils.message("cal.error.error13", shiftName));


                    CalShift o1 = calShift1.deepCopyObj();
                    o1.setExecStartTime(breakStartDate1);
                    o1.setExecEndTime(breakEndDate1);
                    o1.setRestFlag(1);
                    o1.setOperType(1);
                    o1.setShiftName(shiftName);
                    CalShift o2 = calShift2.deepCopyObj();
                    o2.setExecStartTime(breakStartDate2);
                    o2.setExecEndTime(breakEndDate2);
                    o2.setRestFlag(1);
                    o2.setOperType(1);
                    o2.setShiftName(shiftName);
                    calShift.setBreakTimeList(Arrays.asList(o1, o2));
                    continue;
                }


                if (breakTimeList.size() == 1) {
                    CalShift calShift1 = breakTimeList.get(0);
                    String breakStartTime1 = calShift1.getBreakStartTime();
                    String breakEndTime1 = calShift1.getBreakEndTime();
                    Date breakStartDate1 = formatShiftDate(nowDateFormat, breakStartTime1);
                    Date breakEndDate1 = formatShiftDate(nowDateFormat, breakEndTime1);


                    Integer nextDay = calShift1.getNextDay();
                    if (nextDay != null && nextDay == 1) {
                        if (breakEndDate1.after(breakStartDate1)) {
                            breakStartDate1 = formatShiftDate(nextDateFormat, breakStartTime1);
                            breakEndDate1 = formatShiftDate(nextDateFormat, breakEndTime1);
                        } else {
                            breakEndDate1 = formatShiftDate(nextDateFormat, breakEndTime1);
                        }
                    }

                    ExceptionUtil.checkTrueThrowException(shiftStartDate.after(breakStartDate1), shiftName + MessageUtils.message("cal.error.error26"));
                    ExceptionUtil.checkTrueThrowException(breakEndDate1.after(shiftEndDate), shiftName + MessageUtils.message("cal.error.error27"));

                    ExceptionUtil.checkTrueThrowException(!breakEndDate1.after(breakStartDate1), MessageUtils.message("cal.error.error13"));
                    CalShift o = calShift1.deepCopyObj();
                    o.setExecEndTime(breakEndDate1);
                    o.setExecStartTime(breakStartDate1);
                    o.setRestFlag(1);
                    o.setOperType(1);
                    o.setShiftName(shiftName);
                    calShift.setBreakTimeList(Collections.singletonList(o));
                    continue;
                }
            }


        }
    }

    /**
     * 发布排班
     */
    @ApiOperation("排班计划修改完成")
    @PreAuthorize("@ss.hasPermi('mes:cal:calplan:edit')")
    @Log(title = "排班计划修改完成", businessType = BusinessType.UPDATE)
    @PutMapping("/releasePlan")
    public AjaxResult releasePlan(@RequestBody CalPlan calPlan) {
        ExceptionUtil.checkTrueThrowException(!UserConstants.ORDER_STATUS_CONFIRMED.equals(calPlan.getStatus()), MessageUtils.message("cal.error.error17"));

        checkPlan(calPlan);
        calPlan.setUpdateBy(SecurityUtils.getUsername());
        calPlan.setUpdateTime(DateUtils.getNowDate());
        calPlan.setCreateTime(DateUtils.getNowDate());
        calPlan.setCreateBy(SecurityUtils.getUsername());
        calPlanService.releasePlan(calPlan);
        return AjaxResult.success();
    }

    /**
     * 删除排班计划
     */
    @PreAuthorize("@ss.hasPermi('mes:cal:calplan:remove')")
    @Log(title = "排班计划", businessType = BusinessType.DELETE)
    @Transactional
    @DeleteMapping("/{planIds}")
    public AjaxResult remove(@PathVariable Long[] planIds) {
        for (Long planId : planIds) {
            //状态判断
            CalPlan plan = calPlanService.selectCalPlanByPlanId(planId);
            if (!UserConstants.ORDER_STATUS_PREPARE.equals(plan.getStatus())) {
                return AjaxResult.error(MessageUtils.message("cal.error.error18"));
            }
            calShiftService.deleteByPlanId(planId);
            calPlanTeamService.deleteByPlanId(planId);
        }
        return toAjax(calPlanService.deleteCalPlanByPlanIds(planIds));
    }

    /**
     * 导出排班计划列表
     */
    @PreAuthorize("@ss.hasPermi('mes:cal:calplan:export')")
    @Log(title = "排班计划", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, CalPlan calPlan) throws IOException {
        List<CalPlan> list = calPlanService.selectCalPlanList(calPlan);
        List<CalPlanExcelExport> exportList = BeanUtil.copyToList(list, CalPlanExcelExport.class);
        ExcelWriter.write(response, CalPlanExcelExport.class, exportList);
    }

}
