package com.ximai.mes.qc.controller;

import cn.hutool.core.bean.BeanUtil;
import com.ximai.common.annotation.Log;
import com.ximai.common.constant.UserConstants;
import com.ximai.common.core.controller.BaseController;
import com.ximai.common.core.domain.AjaxResult;
import com.ximai.common.core.domain.entity.SysDictData;
import com.ximai.common.core.page.TableDataInfo;
import com.ximai.common.enums.BusinessType;
import com.ximai.common.exception.ServiceException;
import com.ximai.common.utils.DictUtils;
import com.ximai.common.utils.poi.ExcelUtil;
import com.ximai.mes.qc.domain.QcIndex;
import com.ximai.mes.qc.domain.vo.QcIndexExportExcel;
import com.ximai.mes.qc.domain.vo.QcIndexImportExcel;
import com.ximai.mes.qc.domain.vo.QcIndexInputType;
import com.ximai.mes.qc.service.IQcIndexService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.List;

/**
 * 检测项Controller
 *
 * @date 2022-05-17
 */
@RestController
@RequestMapping("/mes/qc/qcindex")
public class QcIndexController extends BaseController {
    @Autowired
    private IQcIndexService qcIndexService;

    /**
     * 查询检测项列表
     */
    @GetMapping("/list")
    public TableDataInfo list(QcIndex qcIndex) {
        startPage();
        List<QcIndex> list = qcIndexService.selectQcIndexList(qcIndex);
        return getDataTable(list);
    }

    /**
     * 导出检测项列表
     */
    @PreAuthorize("@ss.hasPermi('mes:qc:qcindex:export')")
    @Log(title = "检测项", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, QcIndex qcIndex) {
        List<QcIndex> list = qcIndexService.selectQcIndexList(qcIndex);
        ExcelUtil<QcIndexExportExcel> util = new ExcelUtil<QcIndexExportExcel>(QcIndexExportExcel.class);
        List<QcIndexExportExcel> exportList = new ArrayList<QcIndexExportExcel>();
        list.forEach(s -> {
            QcIndexExportExcel temp = new QcIndexExportExcel();
            BeanUtil.copyProperties(s, temp);
            temp.setInputType(QcIndexInputType.get(s.getInputType()).getText());
            List<SysDictData> tempDicList = DictUtils.getDictCache(IQcIndexService.MES_INDEX_TYPE);
            assert tempDicList != null;
            SysDictData dicData = tempDicList.stream().filter(dic -> s.getIndexType().equals(dic.getDictValue())).findFirst().orElseThrow(() -> new ServiceException(String.format("未找到分类：%s", s.getIndexType())));
            temp.setIndexType(dicData.getDictLabel());
            exportList.add(temp);
        });
        util.exportExcel(response, exportList, "检测项数据");
    }

    /**
     * 获取检测项详细信息
     */
    @PreAuthorize("@ss.hasPermi('mes:qc:qcindex:query')")
    @GetMapping(value = "/{indexId}")
    public AjaxResult getInfo(@PathVariable("indexId") Long indexId) {
        return AjaxResult.success(qcIndexService.selectQcIndexByIndexId(indexId));
    }

    /**
     * 新增检测项
     */
    @PreAuthorize("@ss.hasPermi('mes:qc:qcindex:add')")
    @Log(title = "检测项", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody QcIndex qcIndex) {
        if (UserConstants.NOT_UNIQUE.equals(qcIndexService.checkIndexCodeUnique(qcIndex))) {
            return AjaxResult.error("检测项编号已存在！");
        }
        if (UserConstants.NOT_UNIQUE.equals(qcIndexService.checkIndexNameUnique(qcIndex))) {
            return AjaxResult.error("检测项名称已存在！");
        }
        return toAjax(qcIndexService.insertQcIndex(qcIndex));
    }

    /**
     * 修改检测项
     */
    @PreAuthorize("@ss.hasPermi('mes:qc:qcindex:edit')")
    @Log(title = "检测项", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody QcIndex qcIndex) {
        if (UserConstants.NOT_UNIQUE.equals(qcIndexService.checkIndexCodeUnique(qcIndex))) {
            return AjaxResult.error("检测项编号已存在！");
        }
        if (UserConstants.NOT_UNIQUE.equals(qcIndexService.checkIndexNameUnique(qcIndex))) {
            return AjaxResult.error("检测项名称已存在！");
        }
        return toAjax(qcIndexService.updateQcIndex(qcIndex));
    }

    /**
     * 删除检测项
     */
    @PreAuthorize("@ss.hasPermi('mes:qc:qcindex:remove')")
    @Log(title = "检测项", businessType = BusinessType.DELETE)
    @DeleteMapping("/{indexIds}")
    public AjaxResult remove(@PathVariable Long[] indexIds) {
        return toAjax(qcIndexService.deleteQcIndexByIndexIds(indexIds));
    }

    @PostMapping("/importTemplate")
    public void importTemplate(HttpServletResponse response) {
        ExcelUtil<QcIndexImportExcel> util = new ExcelUtil<QcIndexImportExcel>(QcIndexImportExcel.class);
        util.importTemplateExcel(response, "检测项");
    }


    @Log(title = "检测项", businessType = BusinessType.IMPORT)
    @PreAuthorize("@ss.hasPermi('mes:qc:qcindex:import')")
    @PostMapping("/importData")
    public AjaxResult importData(MultipartFile file, boolean updateSupport) throws Exception {
        String operName = getUsername();
        qcIndexService.importExcel(file, updateSupport, operName);
        return AjaxResult.success();
    }
}
