package com.ximai.mes.md.controller;

import com.ximai.common.annotation.Log;
import com.ximai.common.core.controller.BaseController;
import com.ximai.common.core.domain.AjaxResult;
import com.ximai.common.core.page.TableDataInfo;
import com.ximai.common.enums.BusinessType;
import com.ximai.common.utils.poi.ExcelUtil;
import com.ximai.mes.md.domain.MdFactory;
import com.ximai.mes.md.service.IMdFactoryService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * 工厂Controller
 *
 * @author yinjinlu
 * @date 2024-01-18
 */
@RestController
@RequestMapping("/md/mdfactory")
public class MdFactoryController extends BaseController {
    @Autowired
    private IMdFactoryService mdFactoryService;

    /**
     * 查询工厂列表
     */
    @PreAuthorize("@ss.hasPermi('md:mdfactory:list')")
    @GetMapping("/list")
    public TableDataInfo list(MdFactory mdFactory) {
        startPage();
        List<MdFactory> list = mdFactoryService.selectMdFactoryList(mdFactory);
        return getDataTable(list);
    }

    /**
     * 导出工厂列表
     */
    @PreAuthorize("@ss.hasPermi('md:mdfactory:export')")
    @Log(title = "工厂", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, MdFactory mdFactory) {
        List<MdFactory> list = mdFactoryService.selectMdFactoryList(mdFactory);
        ExcelUtil<MdFactory> util = new ExcelUtil<MdFactory>(MdFactory.class);
        util.exportExcel(response, list, "工厂数据");
    }

    /**
     * 获取工厂详细信息
     */
    @PreAuthorize("@ss.hasPermi('md:mdfactory:query')")
    @GetMapping(value = "/{factoryId}")
    public AjaxResult getInfo(@PathVariable("factoryId") Long factoryId) {
        return AjaxResult.success(mdFactoryService.selectMdFactoryByFactoryId(factoryId));
    }

    /**
     * 新增工厂
     */
    @PreAuthorize("@ss.hasPermi('md:mdfactory:add')")
    @Log(title = "工厂", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody MdFactory mdFactory) {
        mdFactory.setCreateBy(getUsername());
        return toAjax(mdFactoryService.insertMdFactory(mdFactory));
    }

    /**
     * 修改工厂
     */
    @PreAuthorize("@ss.hasPermi('md:mdfactory:edit')")
    @Log(title = "工厂", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody MdFactory mdFactory) {
        mdFactory.setUpdateBy(getUsername());
        return toAjax(mdFactoryService.updateMdFactory(mdFactory));
    }

    /**
     * 删除工厂
     */
    @PreAuthorize("@ss.hasPermi('md:mdfactory:remove')")
    @Log(title = "工厂", businessType = BusinessType.DELETE)
    @DeleteMapping("/{factoryIds}")
    public AjaxResult remove(@PathVariable Long[] factoryIds) {
        return toAjax(mdFactoryService.deleteMdFactoryByFactoryIds(factoryIds));
    }
}
