package com.ximai.mes.qc.controller;

import cn.hutool.core.collection.CollUtil;
import com.ximai.common.annotation.Log;
import com.ximai.common.constant.UserConstants;
import com.ximai.common.core.controller.BaseController;
import com.ximai.common.core.domain.AjaxResult;
import com.ximai.common.core.page.TableDataInfo;
import com.ximai.common.enums.BusinessType;
import com.ximai.common.utils.data.ExceptionUtil;
import com.ximai.mes.qc.domain.*;
import com.ximai.mes.qc.service.*;
import com.ximai.mes.wm.service.IWmItemRecptLineService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.List;

/**
 * 来料检验单Controller
 *
 * @date 2022-05-19
 */
@RestController
@RequestMapping("/mes/qc/iqc")
public class QcIqcController extends BaseController {
    @Autowired
    private IQcIqcService qcIqcService;


    @Autowired
    private IQcTemplateProductService qcTemplateProductService;

    @Autowired
    private IQcTemplateIndexService qcTemplateIndexService;

    @Autowired
    private IQcIqcLineService qcIqcLineService;

    @Autowired
    private IQcDefectRecordService qcDefectRecordService;

    @Autowired
    private IWmItemRecptLineService wmItemRecptLineService;


    /**
     * 查询来料检验单列表
     */
    @GetMapping("/list")
    public TableDataInfo list(QcIqc qcIqc) {
        startPage();
        List<QcIqc> list = qcIqcService.selectQcIqcList(qcIqc);
        return getDataTable(list);
    }
    /**
     * 获取来料检验单详细信息
     */
    @PreAuthorize("@ss.hasPermi('mes:qc:iqc:query')")
    @GetMapping(value = "/{iqcId}")
    public AjaxResult getInfo(@PathVariable("iqcId") Long iqcId) {
        return AjaxResult.success(qcIqcService.selectQcIqcByIqcId(iqcId));
    }

    /**
     * 新增来料检验单
     */
    @PreAuthorize("@ss.hasPermi('mes:qc:iqc:add')")
    @Log(title = "来料检验单", businessType = BusinessType.INSERT)
    @Transactional
    @PostMapping
    public AjaxResult add(@RequestBody QcIqc qcIqc) {
        if (UserConstants.NOT_UNIQUE.equals(qcIqcService.checkIqcCodeUnique(qcIqc))) {
            return AjaxResult.error("单据编号已存在！");
        }

        QcTemplateProduct param = new QcTemplateProduct();
        param.setItemId(qcIqc.getItemId());
        List<QcTemplateProduct> templates = qcTemplateProductService.selectQcTemplateProductList(param);
        if (CollUtil.isNotEmpty(templates)) {
            qcIqc.setTemplateId(templates.get(0).getTemplateId());
        } else {
            return AjaxResult.error("当前产品未配置检测模板！");
        }
        qcIqc.setInspector(getUsername());
        qcIqcService.insertQcIqc(qcIqc);
        generateLine(qcIqc);
        Long iqcId = qcIqc.getIqcId();
        return AjaxResult.success(iqcId); //将生成的ID返回给页面
    }

    /**
     * 修改来料检验单
     */
    @PreAuthorize("@ss.hasPermi('mes:qc:iqc:edit')")
    @Log(title = "来料检验单", businessType = BusinessType.UPDATE)
    @Transactional
    @PutMapping
    public AjaxResult edit(@RequestBody QcIqc qcIqc) {
        for (QcIqcLine qcIqcLine : qcIqc.getQcIqcLineList()) {
            qcIqcLineService.updateQcIqcLine(qcIqcLine);
        }

        qcIqc.setInspector(getUsername());

        return toAjax(qcIqcService.updateQcIqc(qcIqc));
    }

    /**
     * 删除来料检验单
     */
    @PreAuthorize("@ss.hasPermi('mes:qc:iqc:remove')")
    @Log(title = "来料检验单", businessType = BusinessType.DELETE)
    @Transactional
    @DeleteMapping("/{iqcIds}")
    public AjaxResult remove(@PathVariable Long[] iqcIds) {
        for (Long iqcId : iqcIds
        ) {
            QcIqc iqc = qcIqcService.selectQcIqcByIqcId(iqcId);
            if (!UserConstants.ORDER_STATUS_PREPARE.equals(iqc.getStatus())) {
                return AjaxResult.error("只能删除草稿状态单据！");
            }
            qcIqcLineService.deleteByIqcId(iqcId);
            QcDefectRecord p2 = new QcDefectRecord();
            p2.setQcId(iqcId);
            p2.setQcType(UserConstants.QC_TYPE_IQC);
            qcDefectRecordService.deleteByQcIdAndType(p2);//删除对应的缺陷记录
        }
        return toAjax(qcIqcService.deleteQcIqcByIqcIds(iqcIds));
    }

    /**
     * 根据头信息生成行信息
     *
     * @param iqc
     */
    private void generateLine(QcIqc iqc) {
        QcTemplateIndex param = new QcTemplateIndex();
        param.setTemplateId(iqc.getTemplateId());
        List<QcTemplateIndex> indexs = qcTemplateIndexService.selectQcTemplateIndexList(param);
        if (CollUtil.isNotEmpty(indexs)) {
            for (QcTemplateIndex index : indexs
            ) {
                QcIqcLine line = new QcIqcLine();
                line.setIqcId(iqc.getIqcId());
                line.setIndexId(index.getIndexId());
                line.setIndexCode(index.getIndexCode());
                line.setIndexName(index.getIndexName());
                line.setIndexType(index.getIndexType());
                line.setQcTool(index.getQcTool());
                line.setCheckMethod(index.getCheckMethod());
                line.setStanderVal(index.getStanderVal());
                line.setUnitOfMeasure(index.getUnitOfMeasure());
                line.setThresholdMax(index.getThresholdMax());
                line.setThresholdMin(index.getThresholdMin());
                line.setCrQuantity(BigDecimal.ZERO);
                line.setMajQuantity(BigDecimal.ZERO);
                line.setMajQuantity(BigDecimal.ZERO);
                qcIqcLineService.insertQcIqcLine(line);
            }
        }
    }

}
