package com.ximai.mes.od.service.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ximai.common.utils.data.DateUtils;
import com.ximai.mes.od.domain.OdPurchaseOrder;
import com.ximai.mes.od.domain.OdSalesOrder;
import com.ximai.mes.od.domain.OdSalesOrderDetail;
import com.ximai.mes.od.domain.OdSalesOrderDetailItem;
import com.ximai.mes.od.mapper.OdPurchaseOrderMapper;
import com.ximai.mes.od.mapper.OdSalesOrderDetailMapper;
import com.ximai.mes.od.mapper.OdSalesOrderMapper;
import com.ximai.mes.od.service.IOdSalesOrderDetailItemService;
import com.ximai.mes.od.service.IOdSalesOrderService;
import lombok.AllArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * 销售订单Service业务层处理
 *
 * @author yinjinlu
 * @date 2024-01-18
 */
@Service
@AllArgsConstructor
public class OdSalesOrderServiceImpl implements IOdSalesOrderService {

    private final OdSalesOrderMapper odSalesOrderMapper;

    private final OdSalesOrderDetailMapper odSalesOrderDetailMapper;

    private final IOdSalesOrderDetailItemService odSalesOrderDetailItemService;

    private final OdPurchaseOrderMapper odPurchaseOrderMapper;

    /**
     * 查询销售订单
     *
     * @param id 销售订单主键
     * @return 销售订单
     */
    @Override
    public OdSalesOrder selectOdSalesOrderById(Long id) {
        final OdSalesOrder odSalesOrder = odSalesOrderMapper.selectOdSalesOrderById(id);
        final OdSalesOrderDetail odSalesOrderDetail = new OdSalesOrderDetail();
        odSalesOrderDetail.setOrderId(id);
        final List<OdSalesOrderDetail> detailList = odSalesOrderDetailMapper.selectOdSalesOrderDetailList(odSalesOrderDetail);
        odSalesOrder.setDetailList(detailList);
        return odSalesOrder;
    }

    /**
     * 查询销售订单列表
     *
     * @param odSalesOrder 销售订单
     * @return 销售订单
     */
    @Override
    public List<OdSalesOrder> selectOdSalesOrderList(OdSalesOrder odSalesOrder) {
        return odSalesOrderMapper.selectOdSalesOrderList(odSalesOrder);
    }

    /**
     * 新增销售订单
     *
     * @param odSalesOrder 销售订单
     * @return 结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public int insertOdSalesOrder(OdSalesOrder odSalesOrder) {
        odSalesOrder.setCreateTime(DateUtils.getNowDate());
        odSalesOrder.setState("1");
        final int i = odSalesOrderMapper.insertOdSalesOrder(odSalesOrder);
        if (i == 0) return 0;
        saveOdSalesOrderDetail(odSalesOrder);
        OdPurchaseOrder update = odPurchaseOrderMapper.getByNo(odSalesOrder.getPurchaseNo());
        if (update != null) {
            update.setState(String.valueOf(Integer.parseInt(update.getState()) + 1));
            odPurchaseOrderMapper.updateOdPurchaseOrder(update);
        }
        return i;
    }

    private void saveOdSalesOrderDetail(OdSalesOrder odSalesOrder) {
        final List<OdSalesOrderDetail> detailList = odSalesOrder.getDetailList();
        for (OdSalesOrderDetail orderDetail : detailList) {
            orderDetail.setOrderId(odSalesOrder.getId());
            orderDetail.setOrderNo(odSalesOrder.getOrderNo());
            orderDetail.setState("1");
            odSalesOrderDetailMapper.insertOdSalesOrderDetail(orderDetail);
            final List<OdSalesOrderDetailItem> itemList = orderDetail.getItemList();
            for (OdSalesOrderDetailItem item : itemList) {
                item.setDetailId(orderDetail.getId());
            }
            odSalesOrderDetailItemService.saveBatch(itemList);
        }
    }

    /**
     * 修改销售订单
     *
     * @param odSalesOrder 销售订单
     * @return 结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public int updateOdSalesOrder(OdSalesOrder odSalesOrder) {
        odSalesOrder.setUpdateTime(DateUtils.getNowDate());
        final int i = odSalesOrderMapper.updateOdSalesOrder(odSalesOrder);
        if (i > 0) {
            final List<OdSalesOrderDetail> detailList = odSalesOrder.getDetailList();
            for (OdSalesOrderDetail orderDetail : detailList) {
                odSalesOrderDetailMapper.updateOdSalesOrderDetail(orderDetail);
                odSalesOrderDetailItemService.remove(Wrappers.<OdSalesOrderDetailItem>lambdaQuery().eq(OdSalesOrderDetailItem::getDetailId, odSalesOrder.getId()));
                final List<OdSalesOrderDetailItem> itemList = orderDetail.getItemList();
                for (OdSalesOrderDetailItem item : itemList) {
                    item.setDetailId(orderDetail.getId());
                }
                odSalesOrderDetailItemService.saveBatch(itemList);
            }
        }
        return i;
    }

    /**
     * 批量删除销售订单
     *
     * @param ids 需要删除的销售订单主键
     * @return 结果
     */
    @Override
    public int deleteOdSalesOrderByIds(String[] ids) {
        odSalesOrderDetailMapper.deleteByOrderIds(ids);
        return odSalesOrderMapper.deleteOdSalesOrderByIds(ids);
    }

    /**
     * 删除销售订单信息
     *
     * @param id 销售订单主键
     * @return 结果
     */
    @Override
    public int deleteOdSalesOrderById(String id) {
        return odSalesOrderMapper.deleteOdSalesOrderById(id);
    }
}
