package com.ximai.mes.md.controller;

import cn.hutool.core.bean.BeanUtil;
import com.ximai.common.annotation.Log;
import com.ximai.common.constant.UserConstants;
import com.ximai.common.core.controller.BaseController;
import com.ximai.common.core.domain.AjaxResult;
import com.ximai.common.core.page.TableDataInfo;
import com.ximai.common.enums.BusinessType;
import com.ximai.common.utils.excel.ExcelWriter;
import com.ximai.mes.md.domain.MdVendor;
import com.ximai.mes.md.dto.MdVendorExcelExport;
import com.ximai.mes.md.dto.MdVendorExcelImport;
import com.ximai.mes.md.service.IMdVendorService;
import com.ximai.mes.wm.utils.WmBarCodeUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;

/**
 * 供应商Controller
 *
 * @date 2022-05-06
 */
@RestController
@RequestMapping("/mes/md/vendor")
public class MdVendorController extends BaseController {
    @Autowired
    private IMdVendorService mdVendorService;

    @Autowired
    private WmBarCodeUtil barCodeUtil;

    /**
     * 查询供应商列表
     */
    @GetMapping("/list")
    public TableDataInfo list(MdVendor mdVendor) {
        startPage();
        List<MdVendor> list = mdVendorService.selectMdVendorList(mdVendor);
        return getDataTable(list);
    }

    /**
     * 导出供应商列表
     */
    @PreAuthorize("@ss.hasPermi('mes:md:vendor:export')")
    @Log(title = "供应商", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, MdVendor mdVendor) throws IOException {
        List<MdVendor> list = mdVendorService.selectMdVendorList(mdVendor);
        List<MdVendorExcelExport> exportList = BeanUtil.copyToList(list, MdVendorExcelExport.class);
        ExcelWriter.write(response, MdVendorExcelExport.class, exportList);
    }

    @PostMapping("/importTemplate")
    public void importTemplate(HttpServletResponse response) throws IOException {
        ExcelWriter.write(response, MdVendorExcelImport.class);
    }


    @Log(title = "供应商", businessType = BusinessType.IMPORT)
    @PreAuthorize("@ss.hasPermi('system:user:import')")
    @PostMapping("/importData")
    public AjaxResult importData(MultipartFile file, boolean updateSupport) throws Exception {
        mdVendorService.importVendor(file.getInputStream(), updateSupport);
        return AjaxResult.success();
    }

    /**
     * 获取供应商详细信息
     */
    @PreAuthorize("@ss.hasPermi('mes:md:vendor:query')")
    @GetMapping(value = "/{vendorId}")
    public AjaxResult getInfo(@PathVariable("vendorId") Long vendorId) {
        return AjaxResult.success(mdVendorService.selectMdVendorByVendorId(vendorId));
    }

    /**
     * 新增供应商
     */
    @PreAuthorize("@ss.hasPermi('mes:md:vendor:add')")
    @Log(title = "供应商", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody MdVendor mdVendor) {
        if (UserConstants.NOT_UNIQUE.equals(mdVendorService.checkVendorCodeUnique(mdVendor))) {
            return AjaxResult.error("供应商编码已存在！");
        }
        if (UserConstants.NOT_UNIQUE.equals(mdVendorService.checkVendorNameUnique(mdVendor))) {
            return AjaxResult.error("供应商名称已存在！");
        }
        if (UserConstants.NOT_UNIQUE.equals(mdVendorService.checkVendorNickUnique(mdVendor))) {
            return AjaxResult.error("供应商简称已存在！");
        }

        mdVendorService.insertMdVendor(mdVendor);
        barCodeUtil.generateBarCode(UserConstants.BARCODE_TYPE_VENDOR, mdVendor.getVendorId(), mdVendor.getVendorCode(), mdVendor.getVendorName());

        return AjaxResult.success(mdVendor.getVendorId());
    }

    /**
     * 修改供应商
     */
    @PreAuthorize("@ss.hasPermi('mes:md:vendor:edit')")
    @Log(title = "供应商", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody MdVendor mdVendor) {
        if (UserConstants.NOT_UNIQUE.equals(mdVendorService.checkVendorCodeUnique(mdVendor))) {
            return AjaxResult.error("供应商编码已存在！");
        }
        if (UserConstants.NOT_UNIQUE.equals(mdVendorService.checkVendorNameUnique(mdVendor))) {
            return AjaxResult.error("供应商名称已存在！");
        }
        if (UserConstants.NOT_UNIQUE.equals(mdVendorService.checkVendorNickUnique(mdVendor))) {
            return AjaxResult.error("供应商简称已存在！");
        }
        return toAjax(mdVendorService.updateMdVendor(mdVendor));
    }

    /**
     * 删除供应商
     */
    @PreAuthorize("@ss.hasPermi('mes:md:vendor:remove')")
    @Log(title = "供应商", businessType = BusinessType.DELETE)
    @DeleteMapping("/{vendorIds}")
    public AjaxResult remove(@PathVariable Long[] vendorIds) {
        return toAjax(mdVendorService.deleteMdVendorByVendorIds(vendorIds));
    }

    /**
     * 同步ERP数据
     */
    @PreAuthorize("@ss.hasPermi('mes:md:vendor:sync')")
    @Log(title = "供应商同步", businessType = BusinessType.SYNC_DATA)
    @PostMapping("/syncData")
    public AjaxResult syncData() {
        mdVendorService.syncErpData();
        return AjaxResult.success();
    }
}
