package com.ximai.mes.md.controller;

import cn.hutool.core.bean.BeanUtil;
import com.ximai.common.annotation.Log;
import com.ximai.common.constant.UserConstants;
import com.ximai.common.core.controller.BaseController;
import com.ximai.common.core.domain.AjaxResult;
import com.ximai.common.core.page.TableDataInfo;
import com.ximai.common.enums.BusinessType;
import com.ximai.common.utils.data.StringUtils;
import com.ximai.common.utils.excel.ExcelWriter;
import com.ximai.mes.md.domain.MdWorkshop;
import com.ximai.mes.md.domain.MdWorkstation;
import com.ximai.mes.md.dto.MdWorkstationExcelExport;
import com.ximai.mes.md.service.*;
import com.ximai.mes.pro.service.IProProcessService;
import com.ximai.mes.wm.service.IWmStorageAreaService;
import com.ximai.mes.wm.service.IWmStorageLocationService;
import com.ximai.mes.wm.service.IWmWarehouseService;
import com.ximai.mes.wm.utils.WmBarCodeUtil;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;

/**
 * 工作站Controller
 *
 * @date 2022-05-10
 */
@RestController
@RequestMapping("/mes/md/workstation")
public class MdWorkstationController extends BaseController {
    @Autowired
    private IMdWorkstationService mdWorkstationService;

    @Autowired
    private IMdWorkstationMachineService mdWorkstationMachineService;

    @Autowired
    private IMdWorkstationToolService mdWorkstationToolService;

    @Autowired
    private IMdWorkstationWorkerService mdWorkstationWorkerService;

    @Autowired
    private IProProcessService proProcessService;

    @Autowired
    private IMdWorkshopService mdWorkshopService;

    @Autowired
    private IWmWarehouseService wmWarehouseService;

    @Autowired
    private IWmStorageLocationService wmStorageLocationService;

    @Autowired
    private IWmStorageAreaService wmStorageAreaService;

    @Autowired
    private WmBarCodeUtil barCodeUtil;

    /**
     * 查询工作站列表
     */
    @ApiOperation(value = "工作中心查询")
    @GetMapping("/list")
    public TableDataInfo<MdWorkstation> list(MdWorkstation mdWorkstation) {
        startPage();
        List<MdWorkstation> list = mdWorkstationService.selectMdWorkstationList(mdWorkstation);
        return getDataTable(list);
    }


    /**
     * 获取工作站详细信息
     */
    @PreAuthorize("@ss.hasPermi('mes:md:workstation:query')")
    @GetMapping(value = "/{workstationId}")
    public AjaxResult getInfo(@PathVariable("workstationId") Long workstationId) {
        return AjaxResult.success(mdWorkstationService.selectMdWorkstationByWorkstationId(workstationId));
    }

    /**
     * 新增工作站
     */
    @PreAuthorize("@ss.hasPermi('mes:md:workstation:add')")
    @Log(title = "工作站", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody MdWorkstation mdWorkstation) {
        if (UserConstants.NOT_UNIQUE.equals(mdWorkstationService.checkWorkStationCodeUnique(mdWorkstation))) {
            return AjaxResult.error("工作站编码已存在！");
        }
        if (UserConstants.NOT_UNIQUE.equals(mdWorkstationService.checkWorkStationNameUnique(mdWorkstation))) {
            return AjaxResult.error("工作站名称已存在！");
        }

        mdWorkstationService.insertMdWorkstation(mdWorkstation);
//        barCodeUtil.generateBarCode(UserConstants.BARCODE_TYPE_WORKSTATION,mdWorkstation.getWorkstationId(), mdWorkstation.getWorkstationCode(),mdWorkstation.getWorkstationName());
        return AjaxResult.success(mdWorkstation.getWorkstationId());
    }

    /**
     * 修改工作站
     */
    @PreAuthorize("@ss.hasPermi('mes:md:workstation:edit')")
    @Log(title = "工作站", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody MdWorkstation mdWorkstation) {
        if (UserConstants.NOT_UNIQUE.equals(mdWorkstationService.checkWorkStationCodeUnique(mdWorkstation))) {
            return AjaxResult.error("工作站编码已存在！");
        }
        if (UserConstants.NOT_UNIQUE.equals(mdWorkstationService.checkWorkStationNameUnique(mdWorkstation))) {
            return AjaxResult.error("工作站名称已存在！");
        }

        MdWorkshop workshop = mdWorkshopService.selectMdWorkshopByWorkshopId(mdWorkstation.getWorkshopId());
        if (!StringUtils.isNotNull(workshop)) {
            return AjaxResult.error("车间不存在！");
        }

        return toAjax(mdWorkstationService.updateMdWorkstation(mdWorkstation));
    }

    /**
     * 删除工作站
     */
    @PreAuthorize("@ss.hasPermi('mes:md:workstation:remove')")
    @Log(title = "工作站", businessType = BusinessType.DELETE)
    @Transactional
    @DeleteMapping("/{workstationIds}")
    public AjaxResult remove(@PathVariable Long[] workstationIds) {
        for (Long workstationId : workstationIds
        ) {
//            mdWorkstationMachineService.deleteByWorkstationId(workstationId);
//            mdWorkstationToolService.deleteByWorkstationId(workstationId);
            mdWorkstationWorkerService.deleteByWorkstationId(workstationId);
        }
        return toAjax(mdWorkstationService.deleteMdWorkstationByWorkstationIds(workstationIds));
    }

    @ApiOperation(value = "默认工作中心查询")
    @GetMapping("/defaultList")
    public List<MdWorkstation> defaultWorkCenter() {
        List<MdWorkstation> list = mdWorkstationService.selectWorkstationByUserId(super.getLoginUser().getUserId());
        return list;
    }

    /**
     * 导出工作站列表
     */
    @PreAuthorize("@ss.hasPermi('mes:md:workstation:export')")
    @Log(title = "工作站", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, MdWorkstation mdWorkstation) throws IOException {
        List<MdWorkstation> list = mdWorkstationService.selectMdWorkstationList(mdWorkstation);
        List<MdWorkstationExcelExport> listExport = BeanUtil.copyToList(list, MdWorkstationExcelExport.class);
        ExcelWriter.write(response, MdWorkstationExcelExport.class, listExport);
    }
}
