package com.ximai.mes.pro.controller;

import cn.hutool.core.bean.BeanUtil;
import com.ximai.common.annotation.Log;
import com.ximai.common.constant.UserConstants;
import com.ximai.common.core.controller.BaseController;
import com.ximai.common.core.domain.AjaxResult;
import com.ximai.common.core.page.TableDataInfo;
import com.ximai.common.enums.BusinessType;
import com.ximai.common.utils.excel.ExcelWriter;
import com.ximai.mes.pro.domain.ProUsage;
import com.ximai.mes.pro.dto.ProUsageExcelExport;
import com.ximai.mes.pro.service.IProUsageService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;

/**
 * title:
 * author:ZHT
 * date:2024/1/15
 * description:
 */
@RestController
@RequestMapping("/mes/pro/usage")
public class ProUsageController extends BaseController {

    @Autowired
    private IProUsageService proUsageService;

    /**
     * 获取用途列表
     */
    @GetMapping("/list")
    public TableDataInfo list(ProUsage proUsage) {
        startPage();
        List<ProUsage> list = proUsageService.selectProUsageList(proUsage);
        return getDataTable(list);
    }

    /**
     * 查询所有可用的用途
     */
    public AjaxResult listAll() {
        ProUsage proUsage = new ProUsage();
        proUsage.setEnableFlag("Y");
        List<ProUsage> list = proUsageService.selectProUsageList(proUsage);
        return AjaxResult.success(list);
    }

    /**
     * 根据id查询用途详细信息
     */
    @PreAuthorize("@ss.hasPermi('mes:pro:usage:query')")
    @GetMapping(value = "/{usageId}")
    public AjaxResult getInfo(@PathVariable("usageId") Long usageId) {
        return AjaxResult.success(proUsageService.selectProUsageByUsageId(usageId));
    }

    /**
     * 新增生产用途
     */
    @PreAuthorize("@ss.hasPermi('mes:pro:usage:add')")
    @Log(title = "生产用途", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody ProUsage proUsage) {
        {
            if (UserConstants.NOT_UNIQUE.equals(proUsageService.checkUsageCodeUnique(proUsage))) {
                return AjaxResult.error("工序编码已存在！");
            }
            if (UserConstants.NOT_UNIQUE.equals(proUsageService.checkUsageNameUnique(proUsage))) {
                return AjaxResult.error("工序名称已存在！");
            }
            return toAjax(proUsageService.insertProUsage(proUsage));
        }
    }

    /**
     * 修改生产用途
     */
    @PreAuthorize("@ss.hasPermi('mes:pro:usage:edit')")
    @Log(title = "生产用途", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody ProUsage proUsage){
        if(UserConstants.NOT_UNIQUE.equals(proUsageService.checkUsageCodeUnique(proUsage))){
            return AjaxResult.error("工序编码已存在！");
        }
        if(UserConstants.NOT_UNIQUE.equals(proUsageService.checkUsageNameUnique(proUsage))){
            return AjaxResult.error("工序名称已存在！");
        }
        return toAjax(proUsageService.updateProProcess(proUsage));
    }

    /**
     * 批量删除指定的用途
     */
    @PreAuthorize("@ss.hasPermi('mes:pro:usage:remove')")
    @Log(title = "生产用途", businessType = BusinessType.DELETE)
    @DeleteMapping("/{usageIds}")
    public AjaxResult remove(@PathVariable Long[] usageIds)
    {
        return toAjax(proUsageService.deleteProUsageByUsageIds(usageIds));
    }

    /**
     * 导出生产用途列表
     */
    @PreAuthorize("@ss.hasPermi('mes:pro:usage:export')")
    @Log(title = "生产用途", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, ProUsage proUsage) throws IOException {
        List<ProUsage> list = proUsageService.selectProUsageList(proUsage);
        List<ProUsageExcelExport> exportList = BeanUtil.copyToList(list, ProUsageExcelExport.class);
        ExcelWriter.write(response, ProUsageExcelExport.class, exportList);
    }
}
