import React, { useState } from 'react';
import styles from '../../index.less';
import { Modal } from 'antd';
import OrderInfo from '../../../materialSynthesis/components/efficiency/orderInfo';
import AnnualPlanBar from './annualPlan';
import MonthlyPlanChart from './monthlyPlanChart';
import PunctualChart from './punctualChart';
import {
  queryBatchInfoPunctualityPenetrating,
  queryPlanQuantityPenetratingByKanban,
} from '../../services/api';

type MonthlyPlanProps = KANBAN.QiPingComprehens.WorkOrderMonthlyIndexData;

const Main: React.FC<{ data: MonthlyPlanProps; kanbanId: string | undefined }> = ({
  data,
  kanbanId,
}) => {
  // 趋势图
  const bgTemp = [
    '',
    '上线准时率趋势图',
    '准时下线率趋势图',
    '入库准时率趋势图',
    '周期准时率趋势图',
  ];
  const [visibleBg, setVisibleBg] = useState(false);
  const [chartData, setChartData] = useState([]);
  const [visibleBgType, setVisibleBgType] = useState(0);
  const closeVisibleBg = () => setVisibleBg(false);
  const clickBg = async (type: number) => {
    if (!kanbanId) return;
    const { data: cData } = await queryBatchInfoPunctualityPenetrating({
      type,
      kanbanSettingId: kanbanId,
      boardType: 2,
    });
    setChartData(cData);
    setVisibleBgType(type);
    setVisibleBg(true);
  };

  // 工单明细
  const numTemp = [
    '',
    '上线准时工单明细',
    '准时下线工单明细',
    '入库准时工单明细',
    '周期准时工单明细',
    '历史月未完成工单明细',
    '实际完工工单明细',
  ];
  const [visibleNum, setVisibleNum] = useState(false);
  const [visibleNumType, setVisibleNumType] = useState(0);
  const [batchInfoParams, setBatchInfoParams] = useState({
    current: 1,
    pageSize: 10,
    type: 1,
    boardType: 2,
    kanbanSettingId: '',
  });
  const closeVisibleNum = () => setVisibleNum(false);
  const clickNum = (e: any, type: number) => {
    if (e) {
      e.stopPropagation();
      e.nativeEvent.stopImmediatePropagation();
    }

    if (!kanbanId) return;

    setBatchInfoParams({
      ...batchInfoParams,
      type,
      kanbanSettingId: kanbanId,
    });
    setVisibleNumType(type);
    setVisibleNum(true);
  };

  // 工单年度计划量
  const [visibleAnnualPlan, setVisibleAnnualPlan] = useState(false);
  const [planQuantity, setPlanQuantity] = useState<{ period: string; val: number }[]>([]);
  const openAnnualPlan = async () => {
    if (!kanbanId) return;
    const { data: cData } = await queryPlanQuantityPenetratingByKanban({
      kanbanSettingId: kanbanId,
      boardType: 2,
    });
    setPlanQuantity(cData);
    setVisibleAnnualPlan(true);
  };
  const closeAnnualPlan = () => setVisibleAnnualPlan(false);

  return (
    <>
      <div className={styles.centerTopTitle}>
        <h4 onClick={openAnnualPlan}>月度计划&OTD指标</h4>
        <p>
          月计划达成率：<span>{data.monthPlanConcludeRatio}%</span>
        </p>
      </div>
      <div className={styles.centerTopChart}>
        <MonthlyPlanChart data={data} onOpenModal={clickNum} />
      </div>
      <div className={styles.centerTopItemContainer}>
        <div className={styles.centerTopItem} onClick={() => clickBg(1)}>
          <p>上线准时</p>
          <h2 onClick={(e) => clickNum(e, 1)}>{data.startOnTimeRatio}%</h2>
        </div>
        <div className={styles.centerTopItem} onClick={() => clickBg(2)}>
          <p>下线准时</p>
          <h2 onClick={(e) => clickNum(e, 2)}>{data.endOnTimeRatio}%</h2>
        </div>
        <div className={styles.centerTopItem} onClick={() => clickBg(3)}>
          <p>入库准时</p>
          <h2 onClick={(e) => clickNum(e, 3)}>{data.stockInOnTimeRatio}%</h2>
        </div>
        <div className={styles.centerTopItem} onClick={() => clickBg(4)}>
          <p>周期准时</p>
          <h2 onClick={(e) => clickNum(e, 4)}>{data.periodicAccordRatio}%</h2>
        </div>
        <div className={styles.centerTopItem} style={{ marginRight: '0px' }}>
          <p>历史月待完工</p>
          <h2 onClick={(e) => clickNum(e, 5)}>{data.hisPendingQuantity}</h2>
        </div>
      </div>

      <Modal
        title={bgTemp[visibleBgType]}
        visible={visibleBg}
        footer={null}
        width={'70%'}
        onCancel={closeVisibleBg}
      >
        <PunctualChart data={chartData} />
      </Modal>

      <Modal
        title={numTemp[visibleNumType]}
        visible={visibleNum}
        footer={null}
        width={'70%'}
        onCancel={closeVisibleNum}
      >
        <OrderInfo params={batchInfoParams} />
      </Modal>

      <Modal
        title={'工单年度计划量'}
        visible={visibleAnnualPlan}
        footer={null}
        width={'70%'}
        onCancel={closeAnnualPlan}
      >
        <AnnualPlanBar data={planQuantity} />
      </Modal>
    </>
  );
};

export default Main;
