import React, { useRef } from 'react';
import { useIntl } from 'umi';
import type { ProFormInstance } from '@ant-design/pro-form';
import {
  DrawerForm,
  ProFormDatePicker,
  ProFormDigit,
  ProFormGroup,
  ProFormText,
} from '@ant-design/pro-form';
import serviceClient from '@/services/report/settings/index';
import moment from 'moment';

export type EditFormProps = {
  values: Partial<Report.Settings.ReportOperationSettingDto>;
  visible: boolean;
  onVisibleChange: (visible: boolean) => void;
  onSubmitting: () => Promise<any>;
  onSuccess: (params: any) => Promise<void>;
  onError: (params: any, error: unknown) => Promise<void>;
};

const EditForm: React.FC<EditFormProps> = (props) => {
  const intl = useIntl();
  const formRef = useRef<ProFormInstance>();

  return (
    <DrawerForm
      formRef={formRef}
      title={intl.formatMessage({
        id: 'pages.report.reportOperationSetting.editForm.title',
        defaultMessage: '编辑报表工序设置',
      })}
      width="740px"
      visible={props.visible}
      onVisibleChange={(visible: boolean) => {
        if (visible) {
          formRef?.current?.setFieldsValue(props.values);
        }
        props.onVisibleChange(visible);
      }}
      onFinish={async (value: Report.Settings.ReportOperationSettingDto) => {
        const params = await props.onSubmitting();
        try {
          const reportOperationSettingUpdate: Report.Settings.ReportOperationSettingUpdate = {
            id: props.values.id as string,
            type: props.values.type?.id as string,
            operation: value.operation as string,
            werks: value.werks as string,
            fevor: value.fevor as string,
            month: moment(value.month)?.format('YYYY-MM-DD HH:mm:ss'),
            dailyQuantity: value.dailyQuantity as number,
          };
          await serviceClient.reportOperationSettingService.update(reportOperationSettingUpdate);
          await props.onSuccess(params);
        } catch (error) {
          await props.onError(params, error);
        }
      }}
    >
      <ProFormGroup>
        <ProFormText
          name={['type', 'name']}
          label={'报表类型'}
          readonly={true}
          width="md"
          rules={[{ required: true }]}
        />
        <ProFormText
          name="fevor"
          label={'生产管理员'}
          width="md"
          readonly={true}
          rules={[{ required: true, message: '请输入生产管理员。' }]}
        />
        <ProFormText
          name={'operation'}
          label={'工序'}
          readonly={true}
          width="md"
          rules={[{ required: true }]}
        />
        <ProFormText
          name="werks"
          label={'工厂'}
          width="md"
          readonly={true}
          rules={[{ required: true, message: '请输入工厂。' }]}
        />
        <ProFormDatePicker.Month
          name="month"
          label={'月份'}
          /*fieldProps={{
            format: 'YYYY-MM',
          }}*/
          width="md"
          readonly={true}
          rules={[{ required: true, message: '请输入月份。' }]}
        />
        <ProFormDigit
          width="md"
          name="dailyQuantity"
          label={'日标准产量'}
          initialValue={0}
          rules={[{ required: true, message: '请输入日标准产量。' }]}
        />
      </ProFormGroup>
    </DrawerForm>
  );
};

export default EditForm;
