import { useEffect, useRef } from 'react';
import * as echarts from 'echarts/lib/echarts';
import { ECharts, getInstanceByDom } from 'echarts';
import { getIntl } from 'umi';
type MonthlyPlanChartProps = {
  data: any;
  onOpenModal: (e: undefined, type: number) => void;
};

const formatOptions = (data: any) => {
  const intl = getIntl();
  const option = {
    color: [
      {
        type: 'linear',
        x: 0,
        y: 0,
        x2: 0,
        y2: 1,
        colorStops: [
          {
            offset: 1,
            color: '#002a54', // 100% 处的颜色
          },
          {
            offset: 0,
            color: '#00ced3', // 0% 处的颜色
          },
        ],
        global: false, // 缺省为 false
      },
      {
        type: 'linear',
        x: 0,
        y: 0,
        x2: 0,
        y2: 1,
        colorStops: [
          {
            offset: 1,
            color: '#002a54',
          },
          {
            offset: 0,
            color: '#3ba7cc',
          },
        ],
        global: false, // 缺省为 false
      },
      {
        type: 'linear',
        x: 0,
        y: 0,
        x2: 0,
        y2: 1,
        colorStops: [
          {
            offset: 1,
            color: '#002a54',
          },
          {
            offset: 0,
            color: '#d1040a',
          },
        ],
        global: false, // 缺省为 false
      },
    ],
    legend: {
      show: true,
      orient: 'vertical',
      right: 5,
      selectedMode: false,
      itemWidth: 10,
      itemHeight: 10,
      textStyle: {
        color: '#fff',
        fontSize: 16,
      },
    },
    grid: {
      left: 20,
      right: 120,
      top: 20,
      bottom: 20,
    },
    xAxis: {
      type: 'category',
      show: false,
      axisLabel: {
        color: '#fff',
        fontSize: 18,
      },
      axisTick: {
        show: false,
      },
      axisLine: {
        show: false,
      },
    },
    yAxis: {
      type: 'value',
      show: false,
    },
    series: [],
  };

  const temp = {
    type: 'bar',
    name: '',
    barGap: '90%',
    label: {
      show: true,
      position: 'top',
      color: '#fff',
    },
    barWidth: 140,
    data: [],
  };

  option.series.push({ ...temp, name: intl.formatMessage({id: '月计划'}), data: [data.monthPlanQuantity] });
  option.series.push({ ...temp, name: intl.formatMessage({id: '实际完工'}), data: [data.actualFinishQuantity] });
  option.series.push({ ...temp, name: intl.formatMessage({id: '超期完工'}), data: [data.expireFinishQuantity] });
  return option;
};

const MonthlyPlanChart: React.FC<MonthlyPlanChartProps> = ({ data, onOpenModal }) => {
  const chartRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    // Initialize chart
    let chart: ECharts | undefined;
    if (chartRef.current !== null) {
      chart = echarts.init(chartRef.current);
    }

    // Add chart resize listener
    // ResizeObserver is leading to a bit janky UX
    function resizeChart() {
      chart?.resize();
    }
    window.addEventListener('resize', resizeChart);

    // Return cleanup function
    return () => {
      chart?.off('click');
      chart?.dispose();
      window.removeEventListener('resize', resizeChart);
    };
  }, []);

  useEffect(() => {
    if (chartRef.current !== null) {
      const chart = getInstanceByDom(chartRef.current);
      const option = formatOptions(data);
      chart?.setOption(option);
      chart?.off('click');
      chart?.on('click', (p) => {
        if (p.seriesName === '实际完工') {
          onOpenModal(undefined, 6);
        }
      });
    }
  }, [data]);

  return <div ref={chartRef} style={{ height: '310px' }} />;
};

export default MonthlyPlanChart;
