import React, { useRef } from 'react';
import { useIntl } from 'umi';
import type { ProFormInstance } from '@ant-design/pro-form';
import {
  ModalForm,
  ProFormGroup,
  ProFormText,
  ProFormDatePicker,
  ProFormDigit,
} from '@ant-design/pro-form';
import serviceClient from '@/services/kanban/settings/index';
import moment from 'moment';

export type CreateFormProps = {
  visible: boolean;
  onVisibleChange: (visible: boolean) => void;
  onSubmitting: () => Promise<any>;
  onSuccess: (params: any) => Promise<void>;
  onError: (params: any, error: unknown) => Promise<void>;
};

const CreateForm: React.FC<CreateFormProps> = (props) => {
  const intl = useIntl();
  const formRef = useRef<ProFormInstance>();

  return (
    <ModalForm
      formRef={formRef}
      title={intl.formatMessage({
        id: 'pages.kanban/settings.qiPingDayNumber.createForm.title',
        defaultMessage: '新增气瓶制造工序日标准产量',
      })}
      width="740px"
      visible={props.visible}
      onVisibleChange={(visible: boolean) => {
        if (visible) {
          formRef?.current?.resetFields();
        }
        props.onVisibleChange(visible);
      }}
      onFinish={async (value: KANBAN.Settings.QiPingDayNumberDto) => {
        const params = await props.onSubmitting();
        try {
          const processDefectRatioDatumCreate: KANBAN.Settings.QiPingDayNumberCreate = {
            ...formRef.current?.getFieldsValue(),
            month: moment(formRef.current?.getFieldValue('month')).format('YYYY-MM-DD HH:mm:ss'),
          };
          await serviceClient.qiPingDayNumberProgressStepService.create(
            processDefectRatioDatumCreate,
          );
          await props.onSuccess(params);
        } catch (error) {
          await props.onError(params, error);
        }
      }}
    >
      <ProFormGroup>
        <ProFormText
          name="werks"
          label={'工厂'}
          width="md"
          rules={[{ required: true, message: '请输入工厂。' }]}
        />
        <ProFormText
          name="fevor"
          label={'生产管理员'}
          width="md"
          rules={[{ required: true, message: '请输入生产管理员。' }]}
        />
        <ProFormText
          name="operationName"
          label={'工序名'}
          width="md"
          rules={[{ required: true, message: '请输入车间。' }]}
        />
        <ProFormDatePicker.Month
          name="month"
          label={'月份'}
          width="md"
          rules={[{ required: true, message: '请输入月份。' }]}
        />
        <ProFormDigit
          name="dayNumber"
          label={'日标准产量'}
          width="md"
          rules={[{ required: true, message: '请输入日标准产量。' }]}
        />
      </ProFormGroup>
    </ModalForm>
  );
};

export default CreateForm;
