import React, { useState } from 'react';
import styles from '../../index.less';
import { Modal } from 'antd';
import AnnualPlanBar from './annualPlan';
import MonthlyPlanChart from './monthlyPlanChart';
import MonthRateChart from './monthRateChart';
import PunctualChart from './punctualChart';
import {
  queryBatchInfoPunctualityPenetrating,
  queryPlanQuantityPenetratingByKanban,
  queryMonthPlanRateData,
} from '../../services/api';
import { useIntl } from 'umi';

type MonthlyPlanProps = KANBAN.ProductionComprehens.WorkOrderMonthlyIndexData;

const Main: React.FC<{ data: MonthlyPlanProps; kanbanId: string | undefined }> = ({
  data,
  kanbanId,
}) => {
  const intl = useIntl();
  // 趋势图
  const bgTemp = [
    '',
    '上线准时率趋势图',
    '准时下线率趋势图',
    '入库准时率趋势图',
    '周期准时率趋势图',
  ];
  const [visibleBg, setVisibleBg] = useState(false);
  const [chartData, setChartData] = useState([]);
  const [visibleBgType, setVisibleBgType] = useState(0);
  const closeVisibleBg = () => setVisibleBg(false);
  const clickBg = async (type: number) => {
    if (!kanbanId) return;
    const { data: cData } = await queryBatchInfoPunctualityPenetrating({
      type,
      kanbanSettingId: kanbanId,
    });
    setChartData(cData);
    setVisibleBgType(type);
    setVisibleBg(true);
  };

  // 工单明细
  const numTemp = [
    '',
    '上线准时工单明细',
    '准时下线工单明细',
    '入库准时工单明细',
    '周期准时工单明细',
    '历史月未完成工单明细',
    '实际完工工单明细',
  ];
  const [visibleNum, setVisibleNum] = useState(false);
  const [visibleNumType, setVisibleNumType] = useState(0);
  const [batchInfoParams, setBatchInfoParams] = useState({
    current: 1,
    pageSize: 10,
    type: 1,
    kanbanSettingId: '',
  });
  const closeVisibleNum = () => setVisibleNum(false);
  const clickNum = (e: any, type: number) => {
    if (e) {
      e.stopPropagation();
      e.nativeEvent.stopImmediatePropagation();
    }

    if (!kanbanId) return;

    setBatchInfoParams({
      ...batchInfoParams,
      type,
      kanbanSettingId: kanbanId,
    });
    setVisibleNumType(type);
    setVisibleNum(true);
  };

  // 工单年度计划量
  const [visibleAnnualPlan, setVisibleAnnualPlan] = useState(false);
  const [planQuantity, setPlanQuantity] = useState<{ period: string; val: number }[]>([]);
  const openAnnualPlan = async () => {
    if (!kanbanId) return;
    const { data: cData } = await queryPlanQuantityPenetratingByKanban({
      kanbanSettingId: kanbanId,
    });
    setPlanQuantity(cData);
    setVisibleAnnualPlan(true);
  };
  const closeAnnualPlan = () => setVisibleAnnualPlan(false);

  const [monthPlanRateData, setMonthPlanRateData] = useState({});
  const [visibleMonth, setVisibleMonth] = useState(false);
  const closeVisibleMonth = () => setVisibleMonth(false);
  const openMonth = async () => {
    if (!kanbanId) return;
    const { data: cData } = await queryMonthPlanRateData({ settingId: kanbanId });
    setMonthPlanRateData(cData);
    setVisibleMonth(true);
  };

  return (
    <>
      <div className={styles.centerTopTitle}>
        <h4 onClick={openAnnualPlan}>{intl.formatMessage({id: '月度计划&OTD指标'})}</h4>
        <p onClick={openMonth}>
        {intl.formatMessage({id: '月计划达成率标'})}：<span>{data.monthPlanConcludeRatio}%</span>
        </p>
      </div>
      <div className={styles.centerTopChart}>
        <MonthlyPlanChart data={data} onOpenModal={clickNum} />
      </div>
      <div className={styles.centerTopItemContainer}>
        <div className={styles.centerTopItem} onClick={() => clickBg(1)}>
          <p>{intl.formatMessage({id: '上线准时'})}</p>
          <h2 onClick={(e) => clickNum(e, 1)}>{data.startOnTimeRatio}%</h2>
        </div>
        <div className={styles.centerTopItem} onClick={() => clickBg(2)}>
          <p>{intl.formatMessage({id: '下线准时'})}</p>
          <h2 onClick={(e) => clickNum(e, 2)}>{data.endOnTimeRatio}%</h2>
        </div>
        <div className={styles.centerTopItem} onClick={() => clickBg(3)}>
          <p>{intl.formatMessage({id: '产品合格'})}</p>
          <h2 onClick={(e) => clickNum(e, 3)}>{data.qualifiedRatio}%</h2>
        </div>
        <div className={styles.centerTopItem} style={{ marginRight: '0px' }}>
          <p>{intl.formatMessage({id: '历史月待完工'})}</p>
          <h2 onClick={(e) => clickNum(e, 5)}>{data.hisPendingQuantity}</h2>
        </div>
      </div>

      <Modal
        title={'月度计划指标'}
        visible={visibleMonth}
        footer={null}
        width={'70%'}
        onCancel={closeVisibleMonth}
      >
        <MonthRateChart data={monthPlanRateData} />
      </Modal>

      <Modal
        title={bgTemp[visibleBgType]}
        visible={visibleBg}
        footer={null}
        width={'70%'}
        onCancel={closeVisibleBg}
      >
        <PunctualChart data={chartData} />
      </Modal>

      <Modal
        title={'工单年度计划量'}
        visible={visibleAnnualPlan}
        footer={null}
        width={'70%'}
        onCancel={closeAnnualPlan}
      >
        <AnnualPlanBar data={planQuantity} />
      </Modal>
    </>
  );
};

export default Main;
