import React, { useEffect, useRef, useState } from 'react';
import * as echarts from 'echarts/lib/echarts';
import type { ECharts } from 'echarts';
import { getInstanceByDom } from 'echarts';

type ProcessOutProps = {
  data: Number;
  setting: any;
};

const formatOptions = (data: any) => {
  const option = {
    tooltip: {
      formatter: "{a} <br/>{b} : {c}%"
    },
    series: [
      {
         name: '利用率',
         type: 'gauge',
         detail: { formatter: '{value}%' },
         data: [{ value: data, name: '利用率' }],
         radius: '110%',
         center: ['50%', '60%'],
         startAngle: 225,
         endAngle: -45,
         splitNumber: 10, // 分割线的数量
         axisLine: {
            lineStyle: {
               width: 18,
               color: [
                  [0.5, '#ff0000'],
                  [0.8, '#fef000'],
                  [1, '#00b58f']
               ]
            }
         },
         axisLabel: {
          color: '#fff'
         },
         axisTick: {
            length: 12,
            lineStyle: {
               color: 'auto',
            }
         },
         splitLine: {
            length: 20,
            lineStyle: {
               color: 'auto',
            }
         },
         pointer: {
            width: 5,
            length: '80%',
            shadowColor: '#ccc', //默认透明
            shadowBlur: 5,
            shadowOffsetX: 3,
            shadowOffsetY: 3
         },
         title: {
            show: true,
            offsetCenter: [0, '-30%'],
            textStyle: {
               color: '#fff',
               fontSize: 30
            }
         },
         detail: {
            show: true,
            offsetCenter: [0, '40%'],
            textStyle: {
               color: 'auto',
               fontSize: 40
            }
         },
         data: [{ value: data, name: '' }],
         animationDuration: 4000, // 仪表盘动画时间
      }
   ]
  };

  return option;
};

const Main: React.FC<ProcessOutProps> = ({ data, setting }) => {
  const chartRef = useRef<HTMLDivElement>(null);
  useEffect(() => {
    // Initialize chart
    let chart: ECharts | undefined;
    if (chartRef.current !== null) {
      chart = echarts.init(chartRef.current);
    }

    // Add chart resize listener
    // ResizeObserver is leading to a bit janky UX
    function resizeChart() {
      chart?.resize();
    }
    window.addEventListener('resize', resizeChart);

    // Return cleanup function
    return () => {
      chart?.off('click');
      chart?.dispose();
      window.removeEventListener('resize', resizeChart);
    };
  }, []);

  useEffect(() => {
    let timer: NodeJS.Timer | null = null;
    // Update chart
    if (chartRef.current !== null) {
      if (timer !== null) clearInterval(timer);
      const chart = getInstanceByDom(chartRef.current);
      const option = formatOptions(data);
      chart?.setOption(option);
    }
    return () => {
      if (timer !== null) clearInterval(timer);
    };
  }, [data, setting]);
  return (
    <>
      <div ref={chartRef} style={{ height: '260px' }} />
    </>
  );
};

export default Main;