import React, { useRef } from 'react';
import { useIntl } from 'umi';
import type { ProFormInstance } from '@ant-design/pro-form';
import { ModalForm, ProFormGroup } from '@ant-design/pro-form';
import { Form, Upload, Button } from 'antd';
import type { RcFile } from 'antd/lib/upload';
import { UploadOutlined } from '@ant-design/icons';

type ExcelExample = {
  download: () => Promise<Blob | null>;
  fileNameProvider: () => Promise<string | null>;
};

export type ImportFormProps = {
  importHandler: (excel: FormData) => Promise<any>;
  visible: boolean;
  onVisibleChange: (visible: boolean) => void;
  onSubmitting: () => Promise<any>;
  onSuccess: (params: any) => Promise<void>;
  onError: (params: any, error: unknown) => Promise<void>;
  excelExample?: ExcelExample;
};

type ExcelUploadProps = {
  value?: RcFile;
  onChange?: (value?: RcFile) => void;
};

const ExcelUpload: React.FC<ExcelUploadProps> = (props) => {
  return (
    <Upload
      name="upload"
      maxCount={1}
      accept="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet, application/vnd.ms-excel"
      beforeUpload={(file) => {
        props.onChange?.(file);
        return false;
      }}
      onRemove={() => {
        props.onChange?.();
      }}
      fileList={props.value ? [props.value] : []}
    >
      <Button icon={<UploadOutlined />}>Upload</Button>
    </Upload>
  );
};

const ImportForm: React.FC<ImportFormProps> = (props) => {
  const intl = useIntl();
  const formRef = useRef<ProFormInstance>();

  return (
    <ModalForm
      formRef={formRef}
      title={intl.formatMessage({
        id: 'component.option.import.excel',
        defaultMessage: '从Excel导入',
      })}
      width="600px"
      visible={props.visible}
      submitter={{
        render: (_, defaultDoms) => {
          return [
            props.excelExample && (
              <Button
                key="abc"
                onClick={async () => {
                  try {
                    const blob = await props.excelExample?.download?.();

                    if (!!!blob) {
                      return;
                    }

                    const link = document.createElement('a');
                    link.style.display = 'none';
                    const url = window.URL.createObjectURL(blob);
                    link.href = url;
                    const fileName = await props.excelExample?.fileNameProvider();
                    if (fileName) {
                      link.download = fileName;
                    }
                    document.body.appendChild(link);
                    const evt = new MouseEvent('click', { bubbles: false, cancelable: false });
                    link.dispatchEvent(evt);

                    document.body.removeChild(link);
                    window.URL.revokeObjectURL(url);
                  } catch (error) {}
                }}
              >
                {intl.formatMessage({
                  id: 'component.option.import.example',
                  defaultMessage: '下载模板',
                })}
              </Button>
            ),
            ...defaultDoms,
          ];
        },
      }}
      onVisibleChange={(visible: boolean) => {
        if (visible) {
          formRef?.current?.setFieldsValue({
            file: undefined,
          });
        }
        props.onVisibleChange(visible);
      }}
      onFinish={async (values: { file: string | Blob }) => {
        const params = await props.onSubmitting();
        try {
          const formData = new FormData();
          formData.append('file', values.file);
          await props.importHandler(formData);
          await props.onSuccess(params);
        } catch (error) {
          await props.onError(params, error);
        }
      }}
    >
      <ProFormGroup>
        <Form.Item
          label={intl.formatMessage({
            id: 'component.option.import.select',
            defaultMessage: '单击选择导入文件',
          })}
          name="file"
          rules={[
            {
              required: true,
              message: intl.formatMessage({
                id: 'component.option.import.file.required',
                defaultMessage: '请选择文件',
              }),
            },
          ]}
        >
          <ExcelUpload />
        </Form.Item>
      </ProFormGroup>
    </ModalForm>
  );
};

export default ImportForm;
