import React, { useRef, useState } from 'react';
import type { ProFormInstance } from '@ant-design/pro-form';
import { DrawerForm, ProFormGroup, ProFormText } from '@ant-design/pro-form';
import ProForm from '@ant-design/pro-form';
import type { ProColumns } from '@ant-design/pro-table';
import { EditableProTable } from '@ant-design/pro-table';
import serviceClient from '@/services/report/settings/index';

export type CreateFormProps = {
  visible: boolean;
  onVisibleChange: (visible: boolean) => void;
  onSubmitting: () => Promise<any>;
  onSuccess: (params: any) => Promise<void>;
  onError: (params: any, error: unknown) => Promise<void>;
};

const CreateForm: React.FC<CreateFormProps> = (props) => {
  const formRef = useRef<ProFormInstance>();

  const [pmEditableKeys, setPmEditableRowKeys] = useState<React.Key[]>(() => []);
  const [pmDataSource, setPmDataSource] = useState<
    Report.Settings.HjProductionManagerProgressStepDto[]
  >(() => []);

  const pmEditColumns: ProColumns<Report.Settings.HjProductionManagerProgressStepDto>[] = [
    {
      title: '工厂',
      dataIndex: 'werks',
      valueType: 'text',
      formItemProps: () => {
        return {
          rules: [{ required: true, message: '此项为必填项' }],
        };
      },
    },
    {
      title: '生产管理员',
      dataIndex: 'fevor',
      valueType: 'text',
      formItemProps: () => {
        return {
          rules: [{ required: true, message: '此项为必填项' }],
        };
      },
    },
    {
      title: '操作',
      valueType: 'option',
    },
  ];
  return (
    <DrawerForm
      formRef={formRef}
      title="新增焊接工序看板配置"
      width="900px"
      visible={props.visible}
      onVisibleChange={(visible: boolean) => {
        if (visible) {
          formRef?.current?.resetFields();
        }
        props.onVisibleChange(visible);
      }}
      onFinish={async () => {
        const formData = formRef?.current?.getFieldsValue();
        const params = await props.onSubmitting();
        try {
          const createDto: Report.Settings.HjKanbanSettingCreate = {
            kanbTitle: formData.kanbTitle,
            werks: formData.werks,
            hjProductionManagerKanbanProgressCreate: {
              hjProductionManagerProgressStepCreates: pmDataSource,
            },
          };
          await serviceClient.hjKanbanSettingService.create(createDto);
          await props.onSuccess(params);
          setPmDataSource([]);

          setPmEditableRowKeys([]);
        } catch (error) {
          await props.onError(params, error);
        }
      }}
    >
      <ProFormGroup>
        <ProFormText name="kanbTitle" label={'看板标题'} width="md" rules={[{ required: true }]} />
        <ProFormText name="werks" label={'工厂'} width="md" rules={[{ required: true }]} />
      </ProFormGroup>
      <ProForm.Item label="生产管理员" name="pmDataSource" trigger="pmOnValuesChange">
        <EditableProTable<Report.Settings.HjProductionManagerProgressStepDto>
          rowKey="rowKey"
          toolBarRender={false}
          columns={pmEditColumns}
          value={pmDataSource}
          recordCreatorProps={{
            newRecordType: 'dataSource',
            position: 'top',
            record: () => ({
              rowKey: Date.now().toString(),
              id: '',
              werks: '',
              fevor: '',
            }),
          }}
          editable={{
            type: 'multiple',
            editableKeys: pmEditableKeys,
            actionRender: (row, _, dom) => {
              return [dom.delete];
            },
            onValuesChange: (record, recordList) => {
              console.log(record);
              console.log(recordList);
              setPmDataSource(recordList);
            },
            onChange: setPmEditableRowKeys,
          }}
        />
      </ProForm.Item>
    </DrawerForm>
  );
};

export default CreateForm;
