import { useEffect, useState } from 'react';
import KanbanStyle from '../components/FHD1080P/Style2';
import SettingForm from './components/SettingForm';
import type { Setting } from './components/SettingForm';
import CountTable from './components/CountTable';
import FinishRate from './components/FinishRate';
import './index.less';
import OutputChart from './components/OutputChart';
import type { CommonParams, StatisTableDto } from './services/api';
import {
  queryQPStatisTableIndex,
  queryQPStatisIndex,
  queryQPStatisQuantityByMonthDay,
  queryQPStatisQuantityByYearMonth,
} from './services/api';

const AirBottleProcessKanban: React.FC = () => {
  const [setting, setSetting] = useState<Setting>({
    autoResize: true,
    intervalTime: 600,
    rowMovingTime: 5,
    QPProgressKanbanSetting: undefined,
    kanbanId: '',
  });
  const [settingFormVisible, setSettingFormVisible] = useState(false);
  const [statisIndex, setStatisIndex] = useState<KANBAN.AirBottleProcessKanban.StatisIndexDto>({
    planNum: 0,
    actualFinishNum: 0,
    handNum: 0,
  });
  const [statisTableData, setStatisTableData] = useState<StatisTableDto>(null);
  const [statisQuantityByMonthDay, setStatisQuantityByMonthDay] = useState([]);
  const [statisQuantityByYearMonth, setStatisQuantityByYearMonth] = useState([]);

  useEffect(() => {
    const { kanbanId, intervalTime, QPProgressKanbanSetting } = setting;
    let t: NodeJS.Timer | null = null;

    if (!kanbanId) {
      if (t !== null) {
        clearInterval(t);
      }
      return;
    }

    const params: CommonParams = {
      worksiteName: QPProgressKanbanSetting?.workLine,
      processConfs: QPProgressKanbanSetting?.qiPingProcessProgressStepList.map((item) => {
        return {
          werks: item.werks,
          fevor: item.fevor,
          operationName: item.extraTypeDes,
        };
      }),
      workUnitConfs: QPProgressKanbanSetting?.qiPingDeviceProgressStepList.map((item) => {
        return {
          ordinal: item.ordinal,
          device: item.device,
          deviceName: item.deviceName,
          operationUnit: item.operationUnit,
        };
      }),
    };

    const requestStatisIndex = async () => {
      const { data } = await queryQPStatisIndex(params);
      setStatisIndex(data);
    };
    const requestStatisTable = async () => {
      const { data } = await queryQPStatisTableIndex(params);
      setStatisTableData(data);
    };
    const requestStatisQuantityByMonthDay = async () => {
      const { data } = await queryQPStatisQuantityByMonthDay(params);
      setStatisQuantityByMonthDay(data);
    };
    const requestStatisQuantityByYearMonth = async () => {
      const { data } = await queryQPStatisQuantityByYearMonth(params);
      setStatisQuantityByYearMonth(data);
    };

    const refresh = () => {
      requestStatisIndex();
      requestStatisTable();
      requestStatisQuantityByMonthDay();
      requestStatisQuantityByYearMonth();
    };

    refresh();

    t = setInterval(() => {
      refresh();
    }, intervalTime * 1000);

    return () => {
      if (t !== null) {
        clearInterval(t);
      }
    };
  }, [setting]);

  return (
    <>
      <KanbanStyle
        headerText="工序看板"
        subTitle={setting.QPProgressKanbanSetting?.kanbTitle}
        autoResize={setting.autoResize}
        onHeaderClick={() => {
          setSettingFormVisible(true);
        }}
      >
        <div className="page-container">
          <div className="wrap">
            <div className="top-container">
              <div className="table-container">
                <div style={{ fontSize: '18px', marginBottom: '20px' }}>
                  手持支数：{statisIndex.handNum || 0}
                </div>
                <CountTable data={statisTableData} setting={setting} />
              </div>
              <div className="count-container">
                <div className="top-count">
                  <div className="left-count">
                    <p>本月计划</p>
                    <h1>{statisIndex.planNum}</h1>
                  </div>
                  <div className="right-count">
                    <p>实际累积</p>
                    <h1>{statisIndex.actualFinishNum || 0}</h1>
                  </div>
                </div>

                <div className="finish-rate">
                  <FinishRate data={statisIndex} />
                </div>
              </div>
            </div>

            <div className="bottom-container">
              <div className="left">
                <OutputChart title="当月产量趋势图" data={statisQuantityByMonthDay} />
              </div>
              <div className="right">
                <OutputChart
                  title="当年产量趋势图"
                  data={statisQuantityByYearMonth}
                  color={'#f2c042'}
                />
              </div>
            </div>
          </div>
        </div>
      </KanbanStyle>
      <SettingForm
        values={{ ...setting }}
        visible={settingFormVisible}
        onVisibleChange={setSettingFormVisible}
        onSubmitting={async (settingParams) => {
          setSetting({ ...settingParams });
        }}
      />
    </>
  );
};

export default AirBottleProcessKanban;
