package com.topsunit.scanservice.ximai.service;

import com.topsunit.scanservice.ximai.common.DateUtil;
import com.topsunit.scanservice.ximai.common.TopsunitException;
import com.topsunit.scanservice.ximai.dao.*;
import com.topsunit.scanservice.ximai.dto.*;
import com.topsunit.scanservice.ximai.dto.mapper.MoctcMapper;
import com.topsunit.scanservice.ximai.entity.*;
import com.topsunit.scanservice.ximai.security.CurrentActor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * Author:   Jack
 * Date:     2021/10/29 11:12
 * Description: 领/退料单服务
 */
@Service
public class MoctcService {
    private final CurrentActor currentActor;
    private final MoctaDao moctaDao;
    private final MoctbDao moctbDao;
    private final MoctcDao moctcDao;
    private final MoctdDao moctdDao;
    private final MocteDao mocteDao;
    private final InvmeDao invmeDao;
    private final InvmcDao invmcDao;
    private final InvmbDao invmbDao;
    private final InvmlDao invmlDao;
    private final CmsniDao cmsniDao;
    private final MoctcMapper moctcMapper;

    public MoctcService(CurrentActor currentActor, MoctaDao moctaDao, MoctbDao moctbDao, MoctcDao moctcDao, MoctdDao moctdDao, MocteDao mocteDao, InvmeDao invmeDao, InvmcDao invmcDao, InvmbDao invmbDao, InvmlDao invmlDao, CmsniDao cmsniDao, MoctcMapper moctcMapper) {
        this.currentActor = currentActor;
        this.moctaDao = moctaDao;
        this.moctbDao = moctbDao;
        this.moctcDao = moctcDao;
        this.moctdDao = moctdDao;
        this.mocteDao = mocteDao;
        this.invmeDao = invmeDao;
        this.invmcDao = invmcDao;
        this.invmbDao = invmbDao;
        this.invmlDao = invmlDao;
        this.cmsniDao = cmsniDao;
        this.moctcMapper = moctcMapper;
    }

    /**
     * 获取可领/退料信息
     *
     * @param criteria
     * @return
     */
    public Optional<MoctcCreateDto> getForCreate(MoctcCreateCriteria criteria) {
        Optional<MoctcCreateDto> moctcCreateDto = moctaDao.findById(moctcMapper.toMoctaId(criteria))
                //.filter(i -> i.getCompany().trim().equals(currentActor.getCompany().trim()))
                .map(moctcMapper::toMoctcCreateDto);
        moctcCreateDto.ifPresent(i -> {
            if (i.getTa030().trim().equals("1")) {// 性质 1.厂内工单、2.委外工单
                i.setTc001("5401");//厂内领料
            } else {
                i.setTc001("5501");//委外领料
            }
            i.setMoctes(moctcMapper.toMocteCreateDtoList(moctbDao.findByTb001AndTb002OrderByTb003AscTb006Asc(criteria.getTa001(), criteria.getTa002()))
                    .stream()
                    .filter(j -> j.getTb004().compareTo(j.getTb005()) > 0)
                    .collect(Collectors.toList()));
            if (i.getMoctes() == null || i.getMoctes().isEmpty()) {
                throw new TopsunitException("工单已领料完成。");
            }
            int ordinal = 1;
            for (int index = 0; index < i.getMoctes().size(); index++) {
                MocteCreateDto j = i.getMoctes().get(index);
                j.setTe003(String.format("%04d", ordinal));
                j.setTe010("");
                invmcDao.findFirstByMc001AndMc002(j.getTb003(), j.getTb009())
                        .ifPresent((x -> {
                            j.setTe025(x.getMc015());
                        }));
                invmbDao.findById(j.getTb003()).ifPresent(x -> {
                    j.setMb022(x.getMb022());
                    j.setIslot(!(x.getMb022().trim().equals("N")));
                });
                ordinal++;
            }
        });
        return moctcCreateDto;
    }

    /**
     * 创建领/退料单
     *
     * @param moctcCreateParams
     */
    @Transactional
    public void create(MoctcCreateParams moctcCreateParams) {
        // 领料信息验证
        List<MocteCreateParams> mocteCreateParamsList = moctcCreateParams.getMoctes();
        if (mocteCreateParamsList == null || mocteCreateParamsList.isEmpty()) {
            throw new TopsunitException("没有任何领料数量。");
        }
        moctcCreateParams.setMoctes(moctcCreateParams
                .getMoctes()
                .stream()
                .filter(i->i.getQuantity().compareTo(BigDecimal.ZERO)>0)
                .collect(Collectors.toList())
        );
        if(moctcCreateParams.getMoctes().size() <= 0){
            return;
        }

        // 工单头
        Mocta mocta = moctaDao.findById(moctcMapper.toMoctaId(moctcCreateParams))
                .orElseThrow(() -> new TopsunitException("工单不存在"));
        // 工单单身
        List<Moctb> moctbs = moctbDao.findByTb001AndTb002OrderByTb003AscTb006Asc(moctcCreateParams.getTa001(), moctcCreateParams.getTa002())
                .stream()
                .filter(j -> j.getTb004().compareTo(j.getTb005()) > 0)
                .collect(Collectors.toList());
        if (moctbs == null || moctbs.isEmpty()) {
            throw new TopsunitException("工单已领料完成");
        }

        // 领/退料单头
        Moctc moctc = initMoctc(moctcCreateParams, mocta);

        // 领/退料工单信息
        Moctd moctd = initMoctd(moctcCreateParams, mocta, moctc);

        // 领/退料单身
        List<MocteCreateParams> exeMocteCreateParamsList = mocteCreateParamsList.stream().filter(i -> i.getQuantity().compareTo(BigDecimal.ZERO) > 0).collect(Collectors.toList());
        List<Mocte> moctes = new ArrayList<>();
        int ordinal = 1;
        for (int index = 0; index < exeMocteCreateParamsList.size(); index++) {
            MocteCreateParams mocteCreateParams = exeMocteCreateParamsList.get(index);
            Optional<Mocte> mocte = initMocte(mocteCreateParams, mocta, moctc, String.format("%04d", ordinal), moctbs);
            if (mocte.isPresent()) {
                moctes.add(mocte.get());
                ordinal++;
            }
        }

        // 合计
        countMocteForMoctc(moctc, moctes);

        // 回写工单
        moctes.forEach(i -> {
            refreshTb005OfMoctb(mocta.getTa001(), mocta.getTa002(), i.getTe004(), i.getTe009(), i.getTe005());
        });

        moctcDao.save(moctc);
        moctdDao.save(moctd);
        moctes.forEach(i -> {
            mocteDao.save(i);
        });
    }

    /**
     * 初始领/退料单头
     *
     * @param moctcCreateParams
     * @param mocta
     * @return
     */
    private Moctc initMoctc(MoctcCreateParams moctcCreateParams, Mocta mocta) {
        Moctc moctc = new Moctc();
        moctc.setTc001(moctcCreateParams.getTc001());
        moctc.setTc002(getNewTC002());
        moctc.setTc003(DateUtil.currentDateString());
        moctc.setTc004(mocta.getTa019());
        moctc.setTc005(mocta.getTa021());
        moctc.setTc006(mocta.getTa032());
        if (mocta.getTa030().trim().equals("1")) {// 性质 1.厂内工单、2.委外工单
            moctc.setTc008("54");// 单据性质 54.厂内领料
        } else {
            moctc.setTc008("55");// 单据性质 55.委外领料
        }
        moctc.setTc014(DateUtil.currentDateString());
        moctc.setTc015(currentActor.getActor().getMv001());
        moctc.setTc021(currentActor.getActor().getMv004());
        moctc.setTc028(currentActor.getActor().getMv001());
        //需合计tc29,tc30
        moctc.setTc031(mocta.getTa083());
        return moctc;
    }

    /**
     * 初始领/退料工单信息
     *
     * @param moctcCreateParams
     * @param mocta
     * @param moctc
     * @return
     */
    private Moctd initMoctd(MoctcCreateParams moctcCreateParams, Mocta mocta, Moctc moctc) {
        Moctd moctd = new Moctd();
        moctd.setTd001(moctc.getTc001());
        moctd.setTd002(moctc.getTc002());
        moctd.setTd003(mocta.getTa001());
        moctd.setTd004(mocta.getTa002());
        moctd.setTd006(BigDecimal.ONE);
        moctd.setTd007(mocta.getTa020());
        moctd.setTd016(DateUtil.currentDateString());
        moctd.setTd026(mocta.getTa083());
        return moctd;
    }

    /**
     * 初始领/退料单身
     *
     * @param mocteCreateParams
     * @param mocta
     * @param moctc
     * @param te003
     * @param moctbs
     * @return
     */
    private Optional<Mocte> initMocte(MocteCreateParams mocteCreateParams, Mocta mocta, Moctc moctc, String te003, List<Moctb> moctbs) {
        Optional<Moctb> moctb = moctbs.stream().filter(i -> i.getTb003().compareTo(mocteCreateParams.getTb003()) == 0 && i.getTb006().compareTo(mocteCreateParams.getTb006()) == 0)
                .findFirst();
        if (!moctb.isPresent()) {
            throw new TopsunitException(String.format("找不到材料品号%s，工艺%s的工单单身。", mocteCreateParams.getTb003(), mocteCreateParams.getTb006()));
        }
        moctb.ifPresent(i -> {
            if (i.getTb004().subtract(i.getTb005()).compareTo(mocteCreateParams.getQuantity()) < 0) {
                throw new TopsunitException(String.format("材料品号%s，工艺%s的领料数量超过可领数量。", mocteCreateParams.getTb003(), mocteCreateParams.getTb006()));
            }
        });

        // 库位验证
        cmsniDao.findById(new CmsniId(moctb.get().getTb009(), mocteCreateParams.getTe025()))
                .orElseThrow(() -> new TopsunitException(String.format("仓库%s库位%s不存在。", moctb.get().getTb009(), mocteCreateParams.getTe025())));

        // 批号验证
        invmbDao.findById(mocteCreateParams.getTb003()).ifPresent(i -> {
            if ((!i.getMb022().equals("N"))  && mocteCreateParams.getTe010().isEmpty()) {
                throw new TopsunitException(String.format("批管理品号%s必须录入批号。", mocteCreateParams.getTb003()));
            }
        });
        invmlDao.findById(new InvmlId(mocteCreateParams.getTb003(), moctb.get().getTb009(), mocteCreateParams.getTe025(), mocteCreateParams.getTe010()))
                .filter(i -> i.getMl005().compareTo(mocteCreateParams.getQuantity()) >= 0)
                .orElseThrow(() -> new TopsunitException(String.format("品号%s批号%s库存不足。", mocteCreateParams.getTb003().trim(), mocteCreateParams.getTe010().trim())));

        Mocte mocte = new Mocte();
        mocte.setTe001(moctc.getTc001());
        mocte.setTe002(moctc.getTc002());
        mocte.setTe003(te003);
        mocte.setTe004(mocteCreateParams.getTb003());
        mocte.setTe005(mocteCreateParams.getQuantity());
        mocte.setTe006(moctb.get().getTb007());
        mocte.setTe008(mocta.getTa020());
        mocte.setTe009(mocteCreateParams.getTb006());
        mocte.setTe010(mocteCreateParams.getTe010());
        mocte.setTe011(mocta.getTa001());
        mocte.setTe012(mocta.getTa002());
        mocte.setTe015(mocta.getTa006());
        mocte.setTe016(moctb.get().getTb011());
        mocte.setTe017(moctb.get().getTb012());
        mocte.setTe018(moctb.get().getTb013());
        mocte.setTe020(mocta.getTa083());
        mocte.setTe025(mocteCreateParams.getTe025());
        mocte.setTe026(moctb.get().getTb007());
        mocte.setTe027(mocteCreateParams.getQuantity());

        return Optional.of(mocte);
    }

    /**
     * 合计
     *
     * @param moctc
     * @param moctes
     */
    private void countMocteForMoctc(Moctc moctc, List<Mocte> moctes) {
        for (Mocte mocte : moctes) {
            moctc.setTc029(moctc.getTc029().add(mocte.getTe005()));
            moctc.setTc030(moctc.getTc030().add(mocte.getTe021()));
        }
    }

    /**
     * 刷新工单单身已领数量
     *
     * @param tb001
     * @param tb002
     * @param tb003
     * @param tb006
     * @param te005
     */
    private void refreshTb005OfMoctb(String tb001, String tb002, String tb003, String tb006, BigDecimal te005) {
        if (tb001 == null || tb001.trim().isEmpty()) {
            return;
        }
        if (tb002 == null || tb002.trim().isEmpty()) {
            return;
        }
        if (tb003 == null || tb003.trim().isEmpty()) {
            return;
        }
        if (tb006 == null || tb006.trim().isEmpty()) {
            return;
        }
        moctbDao.findById(new MoctbId(tb001, tb002, tb003, tb006)).ifPresent(i -> i.setTb005(i.getTb005().add(te005)));
    }

    /**
     * 获取新单号
     *
     * @return
     */
    private String getNewTC002() {
        String prefix = DateUtil.currentDateString();
        return moctcDao.findFirstByTc002StartingWithOrderByTc002Desc(prefix)
                .map(i -> {
                    int currentOrdinal = Integer.parseInt(i.getTc002().replace(prefix, ""));
                    return prefix + String.format("%03d", currentOrdinal + 1);
                })
                .orElse(prefix + "001");
    }
}
