package com.topsunit.scanservice.ximai.security.impl;

import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTDecodeException;
import com.auth0.jwt.exceptions.JWTVerificationException;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.topsunit.scanservice.ximai.common.AppConfig;
import com.topsunit.scanservice.ximai.security.TokenAudience;
import com.topsunit.scanservice.ximai.security.TokenInvalidException;
import com.topsunit.scanservice.ximai.security.TokenManger;
import org.springframework.stereotype.Component;

import java.util.Date;

/**
 * <p>Title: TokenMangerImpl</p>
 * <p>Description: TokenMangerImpl</p>
 *
 * @author xi.feng
 * @version V1.0
 * @date 2021/10/26
 */
@Component
public class TokenMangerImpl implements TokenManger {
    private final AppConfig appConfig;

    public TokenMangerImpl(AppConfig appConfig) {
        this.appConfig = appConfig;
    }

    @Override
    public String generate(String id, String company) {
        Date expiresAt = new Date(System.currentTimeMillis() + appConfig.getTokenExpires());
        return JWT.create()
                .withAudience(id, company)
                .withExpiresAt(expiresAt)
                .sign(Algorithm.HMAC256((id).toString()));
    }

    @Override
    public TokenAudience getTokenAudience(String token) {
        if(token == null || token.isEmpty()) {
            throw new TokenInvalidException();
        }
        String actorId;
        String company;
        try {
            DecodedJWT decode = JWT.decode(token);
            actorId = decode.getAudience().get(0);
            company = decode.getAudience().get(1);
        }
        catch(JWTDecodeException e) {
            throw new TokenInvalidException();
        }
        if(actorId == null || actorId.isEmpty()){
            throw new TokenInvalidException();
        }
        if(company == null || company.isEmpty()){
            throw new TokenInvalidException();
        }

        JWTVerifier jWTVerifier = JWT.require(Algorithm.HMAC256(actorId.toString())).build();
        try {
            jWTVerifier.verify(token);
        }
        catch(JWTVerificationException e) {
            throw new TokenInvalidException(e);
        }
        return TokenAudience.of(actorId, company);
    }
}
