package com.topsunit.scanservice.ximai.service;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.topsunit.scanservice.ximai.common.*;
import com.topsunit.scanservice.ximai.dao.InvmcDao;
import com.topsunit.scanservice.ximai.dao.PurccDao;
import com.topsunit.scanservice.ximai.dao.PurcdDao;
import com.topsunit.scanservice.ximai.dao.PurtdDao;
import com.topsunit.scanservice.ximai.dto.*;
import com.topsunit.scanservice.ximai.dto.mapper.PurccMapper;
import com.topsunit.scanservice.ximai.entity.*;
import com.topsunit.scanservice.ximai.security.CurrentActor;
import com.topsunit.scanservice.ximai.webapi.PrintApi;
import com.topsunit.scanservice.ximai.webapi.dto.PrintField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>Title: PurccService</p>
 * <p>Description: 到货单</p>
 *
 * @author xi.feng
 * @version V1.0
 * @date 2021/10/19
 */
@Service
public class PurccService {
    private final PurccDao purccDao;
    private final PurcdDao purcdDao;
    private final PurtdDao purtdDao;
    private final PurccMapper purccMapper;
    private final CurrentActor currentActor;
    private final PrintApi printApi;
    private final InvmcDao invmcDao;
    @Autowired
    private PrintConfig printConfig;

    public PurccService(PurccDao purccDao, PurcdDao purcdDao, PurtdDao purtdDao, PurccMapper purccMapper, CurrentActor currentActor, PrintApi printApi, InvmcDao invmcDao) {
        this.purccDao = purccDao;
        this.purcdDao = purcdDao;
        this.purtdDao = purtdDao;
        this.purccMapper = purccMapper;
        this.currentActor = currentActor;
        this.printApi = printApi;
        this.invmcDao = invmcDao;
    }

    public Optional<PurccDto> getForPrint(PurccIdCriteria criteria) {
        Optional<PurccDto> purccDto = purccDao
                .findById(purccMapper.toPurccID(criteria))
                //.filter(i -> i.getCompany().trim().equals(currentActor.getCompany().trim()))
                .map(purccMapper::toPurccDto);
        purccDto.ifPresent(i -> i.setPurcds(purccMapper.toPurcdDtoList(purcdDao.findByCd001AndCd002(criteria.getCc001(), criteria.getCc002()))));
        return purccDto;
    }

    public Optional<PurccDto> getForCheck(PurccIdCriteria criteria) {
        Optional<PurccDto> purccDto = purccDao
                .findById(purccMapper.toPurccID(criteria))
                //.filter(i -> i.getCompany().trim().equals(currentActor.getCompany().trim()))
                .filter(i -> i.getCc009().trim().equals("N"))
                .map(purccMapper::toPurccDto);
        purccDto.ifPresent(i ->
                i.setPurcds(
                        purccMapper.toPurcdDtoList(purcdDao.findByCd001AndCd002(criteria.getCc001(), criteria.getCc002()))
                                .stream()
                                .filter(j -> j.getCd034().trim().equals("N"))
                                .collect(Collectors.toList())
                )
        );
        return purccDto;
    }

    @Transactional
    public void check(PurccCheckParams purccCheckParams) {
        Purcc purcc = purccDao
                .findById(purccMapper.toPurccID(purccCheckParams))
                //.filter(i -> i.getCompany().trim().equals(currentActor.getCompany().trim()))
                .filter(i -> i.getCc009().trim().equals("N"))
                .orElseThrow(() -> new TopsunitException("到货单不存在或已检验。"));
        List<Purcd> purcds = purcdDao.findByCd001AndCd002(purccCheckParams.getCc001(), purccCheckParams.getCc002());
        purcds.forEach(i ->
                purccCheckParams.getPurcds().stream()
                        .filter(j -> j.getCd003().equals(i.getCd003()))
                        .findFirst()
                        .ifPresent(x -> {
                            BigDecimal diff = x.getCd018().subtract(i.getCd018());
                            if (diff.compareTo(BigDecimal.ZERO) != 0) {
                                refreshTdc04OfPurtd(i.getCd010(), i.getCd011(), i.getCd012(), diff);
                            }
                            purccMapper.updatePurcd(x, i);
                        }));
        purcds.forEach(i -> {
            int compare = i.getCd018().add(i.getCd019()).compareTo(i.getCd008());
            if (compare > 0) {
                throw new TopsunitException("超出到货数量。");
            }
            if (compare == 0) {
                i.setCd034("Y");
            }
        });
        purcc.setCc003(DateUtil.currentDateString());
        if (purcds.stream().allMatch(i -> i.getCd034().trim().equals("Y"))) {
            purcc.setCc009("Y");
        }
    }

    /**
     * 刷新采购单单身到货数量
     */
    private void refreshTdc04OfPurtd(String td001, String td002, String td003, BigDecimal cd018) {
        if (td001 == null || td001.trim().isEmpty()) {
            return;
        }
        if (td002 == null || td002.trim().isEmpty()) {
            return;
        }
        if (td003 == null || td003.trim().isEmpty()) {
            return;
        }
        purtdDao.findById(new PurtdId(td001, td002, td003)).ifPresent(i -> i.setTdc04(i.getTdc04().add(cd018)));
    }

    public void print(PurccPrintParams printParams) {
        if (printParams == null)
            return;
        List<PurcdPrintParams> purcdPrintParamsList = printParams.getPurcdPrintParamsList();
        if (purcdPrintParamsList == null || purcdPrintParamsList.isEmpty()) {
            return;
        }
        purcdPrintParamsList = printParams.getPurcdPrintParamsList().stream().filter(i -> i.getCount() > 0).collect(Collectors.toList());
        if (purcdPrintParamsList.isEmpty()) {
            return;
        }
        String printCode = Optional.ofNullable(printParams.getUdf01())
                .filter(i -> i.length() > 1)
                .map(i -> i.substring(0, 1))
                .orElse("");
        String url = printConfig.getUrls()
                .stream()
                .filter(i-> i.getCode().equals(printCode))
                .map(PrintConfigUrl::getUrl)
                .findFirst()
                .orElseThrow(()->new TopsunitException("未配置打印机"));

        List<List<PrintField>> codeList = new ArrayList<>();
        purcdPrintParamsList.forEach(i -> {
            purcdDao.findById(new PurcdId(i.getCd001(), i.getCd002(), i.getCd003())).ifPresent(j -> {
                Optional<Purcc> optionalPurcc = purccDao.findById(new PurccId(i.getCd001(), i.getCd002()));

                List<PrintField> objectMap = new ArrayList<>();
                objectMap.add(PrintField.of("DH_Type", j.getCd001().trim()));
                objectMap.add(PrintField.of("DH_No", j.getCd002().trim()));
                objectMap.add(PrintField.of("CG_No", j.getCd011().trim()));
                objectMap.add(PrintField.of("P_No", j.getCd004().trim()));
                objectMap.add(PrintField.of("P_Name", j.getCd005().trim()));
                objectMap.add(PrintField.of("P_Spec", j.getCd006().trim()));
                objectMap.add(PrintField.of("DH_QTY", j.getCd008().toString().trim()));

                // 获取默认库位
                Optional<Invmc> invmcOptional = getDefaultInvmc(j.getCd004());
                if (invmcOptional.isPresent()) {
                    objectMap.add(PrintField.of("DH_KW", invmcOptional.get().getMc015().trim()));
                } else {
                    objectMap.add(PrintField.of("DH_KW", ""));
                }

                // 品号-(空)-单别-单号-序号
                // String qrCode = String.format("%s-%s-%s-%s-%s", j.getCd004().trim(), "", j.getCd001().trim(), j.getCd002().trim(), j.getCd003().trim());
                String qrCode = j.getCd004();
                objectMap.add(PrintField.of("QRCode", qrCode));

                objectMap.add(PrintField.of("CD006", j.getCd006().trim()));
                objectMap.add(PrintField.of("UDF01", j.getUdf01().trim()));
                objectMap.add(PrintField.of("UDF02", j.getUdf02().trim()));
                objectMap.add(PrintField.of("CC010", optionalPurcc
                        .map(Purcc::getCc010)
                        .map(DateUtil::stringToLocalDate)
                        .map(DateUtil::localDateToStandardString)
                        .orElse("")));

                codeList.add(objectMap);

                // 打印数量
                for (Integer printNumber = 1; printNumber < i.getCount(); printNumber++) {
                    codeList.add(objectMap);
                }
            });
        });

        try {
            printApi.print(url, "DH", 1, codeList);
        } catch (JsonProcessingException e) {
            throw new TopsunitException(e.getMessage());
        }
    }

    private Optional<Invmc> getDefaultInvmc(String mc001){
        List<Invmc> byMc001 = invmcDao.findByMc001(mc001)
                .stream()
                .filter(i-> !StringUtil.isNullOrEmpty(i.getMc015()))
                .collect(Collectors.toList());
        if(byMc001.isEmpty()){
            return Optional.empty();
        }
        Optional<Invmc> hasMo013 = byMc001
                .stream()
                .filter(i -> !StringUtil.isNullOrEmpty(i.getMc013()))
                .sorted((x, y) -> y.getMc013().compareTo(x.getMc013()))
                .findFirst();
        if(hasMo013.isPresent()){
            return hasMo013;
        }
        return byMc001.stream().findFirst();
    }
}
