package com.topsunit.scanservice.ximai.service;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.topsunit.scanservice.ximai.common.TopsunitException;
import com.topsunit.scanservice.ximai.dao.CmsniDao;
import com.topsunit.scanservice.ximai.dao.MocthDao;
import com.topsunit.scanservice.ximai.dao.MoctiDao;
import com.topsunit.scanservice.ximai.dao.PurmaDao;
import com.topsunit.scanservice.ximai.dto.*;
import com.topsunit.scanservice.ximai.dto.mapper.MocthMapper;
import com.topsunit.scanservice.ximai.entity.CmsniId;
import com.topsunit.scanservice.ximai.entity.Mocth;
import com.topsunit.scanservice.ximai.entity.Mocti;
import com.topsunit.scanservice.ximai.security.CurrentActor;
import com.topsunit.scanservice.ximai.webapi.PrintApi;
import com.topsunit.scanservice.ximai.webapi.dto.PrintField;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>Title: MocthService</p>
 * <p>Description: 委外进货单</p>
 *
 * @author xi.feng
 * @version V1.0
 * @date 2021/10/27
 */
@Service
public class MocthService {
    private final MocthDao mocthDao;
    private final MoctiDao moctiDao;
    private final PurmaDao purmaDao;
    private final MocthMapper mapper;
    private final CmsniDao cmsniDao;
    private final CurrentActor currentActor;
    private final PrintApi printApi;

    public MocthService(MocthDao mocthDao, MoctiDao moctiDao, PurmaDao purmaDao, MocthMapper mapper, CmsniDao cmsniDao, CurrentActor currentActor, PrintApi printApi) {
        this.mocthDao = mocthDao;
        this.moctiDao = moctiDao;
        this.purmaDao = purmaDao;
        this.mapper = mapper;
        this.cmsniDao = cmsniDao;
        this.currentActor = currentActor;
        this.printApi = printApi;
    }

    public Optional<MocthDto> getForPrint(MocthIdCriteria criteria) {
        Optional<MocthDto> mocthDto = mocthDao
                .findById(mapper.toMocthId(criteria))
                //.filter(i -> i.getCompany().trim().equals(currentActor.getCompany().trim()))
                .map(mapper::toMocthDto)
                .map(i -> {
                    purmaDao.findById(i.getTh005()).ifPresent(j -> i.setMa003(j.getMa003()));
                    return i;
                });
        mocthDto.ifPresent(i ->
                i.setMoctis(
                        moctiDao.findByTi001AndTi002(criteria.getTh001(), criteria.getTh002()).stream()
                                .map(mapper::toMoctiDto)
                                .collect(Collectors.toList())
                ));
        return mocthDto;
    }

    public Optional<MocthDto> getForCheck(MocthIdCriteria criteria) {
        Optional<MocthDto> mocthDto = mocthDao
                .findById(mapper.toMocthId(criteria))
                //.filter(i -> i.getCompany().trim().equals(currentActor.getCompany().trim()))
                .filter(i -> i.getTh023().trim().equals("N"))
                .map(mapper::toMocthDto)
                .map(i -> {
                    purmaDao.findById(i.getTh005()).ifPresent(j -> i.setMa003(j.getMa003()));
                    return i;
                });
        mocthDto.ifPresent(i ->
                i.setMoctis(
                        moctiDao.findByTi001AndTi002(criteria.getTh001(), criteria.getTh002()).stream()
                                .filter(j -> j.getTi037().trim().equals("N"))
                                .map(mapper::toMoctiDto)
                                .collect(Collectors.toList())
                ));
        return mocthDto;
    }

    public Optional<MocthDto> getForPutOn(MocthIdCriteria criteria) {
        Optional<MocthDto> mocthDto = mocthDao
                .findById(mapper.toMocthId(criteria))
                //.filter(i -> i.getCompany().trim().equals(currentActor.getCompany().trim()))
                .filter(i -> i.getTh023().trim().equals("N"))
                .map(mapper::toMocthDto)
                .map(i -> {
                    purmaDao.findById(i.getTh005()).ifPresent(j -> i.setMa003(j.getMa003()));
                    return i;
                });
        mocthDto.ifPresent(i ->
                i.setMoctis(
                        moctiDao.findByTi001AndTi002(criteria.getTh001(), criteria.getTh002()).stream()
                                .filter(j -> j.getTi037().trim().equals("N"))
                                .map(mapper::toMoctiDto)
                                .collect(Collectors.toList())
                ));
        return mocthDto;
    }

    public void print(MocthPrintParams printParams) {
        if (printParams == null)
            return;
        List<MoctiPrintParams> moctis = printParams.getMoctis();
        if (moctis == null || moctis.isEmpty()) {
            return;
        }

        List<List<PrintField>> codeList = new ArrayList<>();
        moctis.stream().forEach(i -> {
            moctiDao.findByTi001AndTi002(printParams.getTh001(), printParams.getTh002()).stream().findFirst().ifPresent(j -> {
                List<PrintField> objectMap = new ArrayList<>();
                objectMap.add(PrintField.of("WWJH_Type", j.getTi001().trim()));
                objectMap.add(PrintField.of("WWJH_No", j.getTi002().trim()));
                objectMap.add(PrintField.of("GD_No", j.getTi014().trim()));
                objectMap.add(PrintField.of("P_No", j.getTi004().trim()));
                objectMap.add(PrintField.of("P_Name", j.getTi005().trim()));
                objectMap.add(PrintField.of("P_Spec", j.getTi006().trim()));
                objectMap.add(PrintField.of("LotNo", j.getTi010().trim()));
                objectMap.add(PrintField.of("WWJH_KW", j.getTi060().trim()));

                // 品号-批号-单别-单号-序号
                String qrCode = String.format("%s-%s-%s-%s-%s", j.getTi004().trim(), j.getTi010().trim(), j.getTi001().trim(), j.getTi002().trim(), j.getTi003().trim());
                objectMap.add(PrintField.of("QRCode", qrCode));

                codeList.add(objectMap);

                // 打印数量
                for (Integer printNumber = 1; printNumber < i.getCount(); printNumber++) {
                    codeList.add(objectMap);
                }
            });
        });

        try {
            printApi.print("WWJH", 1, codeList);
        } catch (JsonProcessingException e) {
            throw new TopsunitException(e.getMessage());
        }
    }

    @Transactional
    public void check(MocthCheckParams checkParams) {
        Mocth mocth = mocthDao
                .findById(mapper.toMocthId(checkParams))
                //.filter(i -> i.getCompany().trim().equals(currentActor.getCompany().trim()))
                .filter(i -> i.getTh023().trim().equals("N"))
                .orElseThrow(() -> new TopsunitException("委外进货单不存在或已审核"));
        List<Mocti> moctis = moctiDao.findByTi001AndTi002(checkParams.getTh001(), checkParams.getTh002());
        moctis.forEach(i -> {
            checkParams.getMoctis().stream()
                    .filter(j -> j.getTi003().equals(i.getTi003()))
                    .findFirst()
                    .ifPresent(j -> {
                        mapper.updateMocti(j, i);
                        i.setTi025(i.getTi020().multiply(i.getTi024()));
                        if (mocth.getTh015().trim().equals("1")) {
                            i.setTi044(i.getTi025().divide(mocth.getTh030().add(BigDecimal.ONE), 2));
                            i.setTi045(i.getTi025().subtract(i.getTi044()));
                        } else if (mocth.getTh015().trim().equals("2")) {
                            i.setTi044(i.getTi025());
                            i.setTi045(i.getTi025().multiply(mocth.getTh030()));
                        } else {
                            i.setTi044(i.getTi025());
                            i.setTi045(BigDecimal.ZERO);
                        }
                        i.setTi046(i.getTi044().multiply(mocth.getTh008()));
                        i.setTi047(i.getTi045().multiply(mocth.getTh008()));
                    });
        });
        if (moctis.stream().anyMatch(i -> i.getTi019().add(i.getTi021()).add(i.getTi022()).compareTo(i.getTi007()) > 0)) {
            throw new TopsunitException("超出进货数量。");
        }
    }

    @Transactional
    public void putOn(MocthPutOnParams putOnParams) {
        Mocth mocth = mocthDao
                .findById(mapper.toMocthId(putOnParams))
                //.filter(i -> i.getCompany().trim().equals(currentActor.getCompany().trim()))
                .filter(i -> i.getTh023().trim().equals("N"))
                .orElseThrow(() -> new TopsunitException("委外进货单不存在或已审核"));
        List<Mocti> moctis = moctiDao.findByTi001AndTi002(putOnParams.getTh001(), putOnParams.getTh002());
        moctis.forEach(i -> {
            putOnParams.getMoctis().stream()
                    .filter(j -> j.getTi003().equals(i.getTi003()))
                    .findFirst()
                    .ifPresent(j -> {
                        cmsniDao.findById(new CmsniId(i.getTi009(), j.getTi060()))
                                //.filter(x -> x.getCompany().trim().equals(currentActor.getCompany().trim()))
                                .orElseThrow(() -> new TopsunitException("库位不存在"));
                        mapper.updateMocti(j, i);
                    });
        });
    }
}
