package com.topsunit.scanservice.ximai.service;

import cn.hutool.core.bean.BeanUtil;
import com.topsunit.scanservice.ximai.common.AppConfig;
import com.topsunit.scanservice.ximai.common.StringUtil;
import com.topsunit.scanservice.ximai.common.TopsunitException;
import com.topsunit.scanservice.ximai.dao.CmsmvDao;
import com.topsunit.scanservice.ximai.dto.ChangePasswordParams;
import com.topsunit.scanservice.ximai.dto.LoginParams;
import com.topsunit.scanservice.ximai.dto.LoginResult;
import com.topsunit.scanservice.ximai.dto.SetPasswordParams;
import com.topsunit.scanservice.ximai.dto.mapper.LoginMapper;
import com.topsunit.scanservice.ximai.entity.Cmsmv;
import com.topsunit.scanservice.ximai.security.CurrentActor;
import com.topsunit.scanservice.ximai.security.Encryptor;
import com.topsunit.scanservice.ximai.security.TokenManger;
import com.topsunit.scanservice.ximai.security.TokenService;
import com.topsunit.scanservice.ximai.webapi.MesLoginService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * <p>Title: CmsmvService</p>
 * <p>Description: 员工基本信息档</p>
 *
 * @author xi.feng
 * @version V1.0
 * @date 2021/10/18
 */
@Service
public class CmsmvService {

    @Autowired
    MesLoginService mesLoginService;

    private final CmsmvDao cmsmvDao;
    private final LoginMapper loginMapper;
    private final TokenManger tokenManger;
    private final CurrentActor currentActor;
    private final Encryptor encryptor;
    private final AppConfig appConfig;
    @Autowired
    private TokenService tokenService;

    public CmsmvService(CmsmvDao cmsmvDao, LoginMapper loginMapper, TokenManger tokenManger, CurrentActor currentActor, Encryptor encryptor, AppConfig appConfig) {
        this.cmsmvDao = cmsmvDao;
        this.loginMapper = loginMapper;
        this.tokenManger = tokenManger;
        this.currentActor = currentActor;
        this.encryptor = encryptor;
        this.appConfig = appConfig;
    }

    public LoginResult login(LoginParams loginParams) {
        return cmsmvDao.findById(loginParams.getUsername())
                .filter(i -> {
                    if (StringUtil.isNullOrEmpty(i.getUdf01())) {
                        if (loginParams.getPassword().equals(appConfig.getDefaultPassword())) {
                            return true;
                        } else {
                            return false;
                        }
                    }
                    return isMatchPassword(loginParams.getPassword(), i.getUdf01());
                })
                .map(loginMapper::toLoginResult)
                .map(i -> {
                    i.setToken(tokenManger.generate(i.getMv001(), loginParams.getCompany()));
                    return i;
                })
                .orElseThrow(() -> new TopsunitException("用户名或密码错误。"));
    }

    public LoginResult loginByMes(LoginParams loginParams) {
        com.topsunit.scanservice.ximai.webapi.dto.LoginParams loginParams2 = BeanUtil.copyProperties(loginParams, com.topsunit.scanservice.ximai.webapi.dto.LoginParams.class);
        com.topsunit.scanservice.ximai.webapi.dto.LoginResult loginResult2 = mesLoginService.login(loginParams2);
        if(loginResult2.getCode()==500){
            throw new TopsunitException(loginResult2.getMsg());
        }
        return tokenService.getUserByToken(loginResult2.getToken());
    }

    @Transactional
    public void setPassword(SetPasswordParams params) {
        Cmsmv cmsmv = cmsmvDao.findById(params.getMv001()).orElseThrow(() -> new TopsunitException("用户不存在"));
        if (!StringUtil.isNullOrEmpty(cmsmv.getUdf01())) {
            throw new TopsunitException("已经设置密码");
        }
        cmsmv.setUdf01(encryptor.hash(params.getPassword()));
    }

    @Transactional
    public void changePassword(ChangePasswordParams params) {
        Cmsmv cmsmv = cmsmvDao.findById(currentActor.getActor().getMv001()).orElseThrow(() -> new TopsunitException("用户不存在"));
        if (StringUtil.isNullOrEmpty(cmsmv.getUdf01())) {
            if (!params.getOldPassWord().equals(appConfig.getDefaultPassword())) {
                throw new TopsunitException("原密码输入有误");
            }
        } else if (!isMatchPassword(params.getOldPassWord(), cmsmv.getUdf01())) {
            throw new TopsunitException("原密码输入有误");
        }
        cmsmv.setUdf01(encryptor.hash(params.getNewPassword()));
    }

    private boolean isMatchPassword(String cleartext, String ciphertext){
        return encryptor.hash(cleartext).equals(ciphertext);
    }
}
