package com.topsunit.scanservice.ximai.common;

import jcifs.smb.SmbFile;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import static com.topsunit.scanservice.ximai.common.FileSystemException.rethrowFileSystemException;

/**
 * 共享文件传输协议
 * @ Author: chicheng.li
 * @ Date  : 2022/1/6
 */

public class SmbFileSystem extends FileSystem {

    String uri;

    @Override
    public boolean exists(String path) throws FileSystemException {
        try{
            SmbFile smbFile = new SmbFile(uri+path);
            return smbFile.exists();
        }catch(Throwable ex){
            return rethrowFileSystemException(ex);
        }
    }

    @Override
    public FileView stat(String path) throws FileSystemException {
        if(!path.startsWith(Path.SEPARATOR)){
            path= Path.SEPARATOR + path;
        }
        try{
            SmbFile smbFile = new SmbFile(uri+path);
            FileView fv = new FileView(smbFile.getPath().replace(uri,""), smbFile.getContentLength(), smbFile.isDirectory(), smbFile.getLastModified(),
                    smbFile.getDate(), smbFile.getType(), null, null, null);
            fv.setName(smbFile.getName());
            return fv;
        }catch(Throwable ex){
            return rethrowFileSystemException(ex);
        }
    }

    @Override
    public FileView[] ls(String path) throws FileSystemException {
        if(!path.startsWith(Path.SEPARATOR)){
            path= Path.SEPARATOR + path;
        }
        try {
            //smb文件夹必须以/结尾
            if(!path.endsWith("/")){
                path=path+"/";
            }
            SmbFile smbFile = new SmbFile(uri+path);
            if (!smbFile.isDirectory()) {
                throw new FileSystemException(String.format("[%s]不是文件夹"));
            }
            SmbFile[] files = smbFile.listFiles();
            FileView[] rst = new FileView[files.length];
            int i=0;
            for(SmbFile nextSmbFile : smbFile.listFiles()){
                FileView fv = new FileView(nextSmbFile.getPath().replace(uri,""), nextSmbFile.getContentLength(), nextSmbFile.isDirectory(), nextSmbFile.getLastModified(),
                        nextSmbFile.getDate(), nextSmbFile.getType(), null, null, null);
                fv.setName(nextSmbFile.getName());
                rst[i] = fv;
                i++;
            }
            return rst;
        }catch(Throwable ex){
            return rethrowFileSystemException(ex);
        }
    }

    @Override
    public boolean mkdirs(String path) throws FileSystemException {
        if(!path.startsWith(Path.SEPARATOR)){
            path= Path.SEPARATOR + path;
        }
        try{
            SmbFile smbFile = new SmbFile(uri+path);
            if(!smbFile.exists()){
                smbFile.mkdirs();
            }
            return true;
        }catch(Throwable ex){
            return rethrowFileSystemException(ex);
        }
    }

    @Override
    public InputStream open(String file) throws FileSystemException {
        if(!file.startsWith(Path.SEPARATOR)){
            file= Path.SEPARATOR +file;
        }
        try{
            SmbFile smbFile = new SmbFile(uri+file);
            return smbFile.getInputStream();
        }catch(Throwable ex){
            return rethrowFileSystemException(ex);
        }
    }

    @Override
    public OutputStream create(String file, boolean override) throws FileSystemException {
        if(!file.startsWith(Path.SEPARATOR)){
            file= Path.SEPARATOR +file;
        }
        try{
            return this.create(file, null, override);
        }catch(Throwable ex){
            return rethrowFileSystemException(ex);
        }
    }

    @Override
    public OutputStream create(String file, String contentType, boolean override) throws FileSystemException {
        try{
            SmbFile smbFile = new SmbFile(uri+file);
            if(override && smbFile.exists()){
                smbFile.delete();
                return smbFile.getOutputStream();
            }else{
                return smbFile.getOutputStream();
            }
        }catch(Throwable ex){
            return rethrowFileSystemException(ex);
        }
    }

    @Override
    public void rm(String path, boolean force, boolean recursive) throws FileSystemException {
        try{
            SmbFile smbFile = new SmbFile(uri+path);
            this.rmExe(smbFile, force, recursive);
        }catch(Throwable ex){
            rethrowFileSystemException(ex);
        }
    }

    /**
     * 删除给定路径
     *
     * @param file      共享文件对象
     * @param force     如果文件夹中存在文件是否强制删除
     * @param recursive 如果文件夹中存在目录是否递归删除
     * @throws FileSystemException
     */
    @SuppressWarnings("unchecked")
    protected void rmExe(SmbFile file, boolean force, boolean recursive) throws IOException {
        try {
            if (file.isDirectory()) {
                SmbFile[] files = file.listFiles();

                if (!force && 0 < files.length) {
                    throw new IOException("You cannot delete non-empty directory, use force=true to overide");
                }

                for (SmbFile f : files) {
                    if (!recursive && f.isDirectory()) {
                        throw new IOException("Directory has contents, cannot delete without recurse=true");
                    }
                    if(f.isDirectory()){
                        this.rmExe(f, force, recursive);
                    }else{
                        f.delete();
                    }
                }
                file.delete();
            } else {
                file.delete();
            }
        } catch (Exception e) {
            throw new IOException(e);
        }
    }

    @Override
    public void rename(String oldPath, String newPath) throws FileSystemException {
        try{
            SmbFile sf = new SmbFile(uri + oldPath);
            sf.renameTo(new SmbFile(uri + newPath));
        }catch(Throwable ex){
            rethrowFileSystemException(ex);
        }
    }

    @Override
    public void disconnect() throws FileSystemException {

    }

    public String getUri() {
        return uri;
    }

    public void setUri(String uri) {
        this.uri = uri;
    }

}
