package com.topsunit.scanservice.ximai.service;

import com.topsunit.scanservice.ximai.dao.InvmaDao;
import com.topsunit.scanservice.ximai.dao.InvmbDao;
import com.topsunit.scanservice.ximai.dao.InvmcDao;
import com.topsunit.scanservice.ximai.dao.InvmlDao;
import com.topsunit.scanservice.ximai.dto.*;
import com.topsunit.scanservice.ximai.dto.mapper.InvmbMapper;
import com.topsunit.scanservice.ximai.entity.*;
import jdk.nashorn.internal.parser.DateParser;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.RequestBody;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.math.BigDecimal;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * <p>Title: InvmbService</p>
 * <p>Description: 品号基本信息档</p>
 *
 * @author xi.feng
 * @version V1.0
 * @date 2021/11/11
 */
@Service
public class InvmbService {

    @Autowired
    private InvmaDao invmaDao;
    @Autowired
    private InvmcDao invmcDao;
    private final InvmbDao invmbDao;
    private final InvmlDao invmlDao;
    private final InvmbMapper invmbMapper;

    public InvmbService(InvmbDao invmbDao, InvmlDao invmlDao, InvmbMapper invmbMapper) {
        this.invmbDao = invmbDao;
        this.invmlDao = invmlDao;
        this.invmbMapper = invmbMapper;
    }

    public Optional<InvmbDto> getForInvmbInfo(InvmbIdCriteria criteria) {
        return invmbDao.findById(criteria.getMb001())
                .map(invmbMapper::toInvmbDto)
                .map(i -> {
                    i.setInvmls(invmbMapper.toInvmlDto(
                            invmlDao.findByMl001OrderByMl002AscMl003AscMl004Asc(criteria.getMb001())
                                    .stream()
                                    .filter(j -> j.getMl005().compareTo(BigDecimal.ZERO) != 0)
                                    .collect(Collectors.toList())
                    ));
                    return i;
                });
    }

    public List<InvmbDto> getInvmbList(InvmbCriteria criteria) {
        Specification<Invmb> query = new Specification<Invmb>() {
            @Override
            public Predicate toPredicate(Root<Invmb> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) {
                List<Predicate> predicates = new ArrayList<Predicate>();
                if (StringUtils.isNotBlank(criteria.getMb001())) {
                    Predicate predicate = criteriaBuilder.like(root.get("mb001"), criteria.getMb001() + "%");
                    predicates.add(predicate);
                }
                if (StringUtils.isNotBlank(criteria.getMb109())) {
                    Predicate predicate = criteriaBuilder.like(root.get("mb109"), criteria.getMb109() + "%");
                    predicates.add(predicate);
                }
                if (StringUtils.isNotBlank(criteria.getNeUdf08())) {
                    Predicate predicate = criteriaBuilder.notEqual(root.get("udf08"), criteria.getNeUdf08());
                    predicates.add(predicate);
                }
                if (StringUtils.isNotBlank(criteria.getGtEqCreateDate())) {
                    Predicate predicate = criteriaBuilder.greaterThanOrEqualTo(root.get("createDate"), criteria.getGtEqCreateDate());
                    predicates.add(predicate);
                }
                if (StringUtils.isNotBlank(criteria.getGtEqModiDate())) {
                    Predicate predicate = criteriaBuilder.greaterThanOrEqualTo(root.get("modiDate"), criteria.getGtEqModiDate());
                    predicates.add(predicate);
                }
                return criteriaQuery.where(predicates.toArray(new Predicate[0])).getRestriction();
            }
        };
        Page<Invmb> list = invmbDao.findAll(query, criteria.toPageable());
        List<InvmbDto> rstList = invmbMapper.toInvmbDtoList(list.toList());
        List<String> mb001List = rstList.stream().map(s->s.getMb001()).collect(Collectors.toList());
        Map<String, List<Invmc>> invmcMap = invmcDao.findByMc001In(mb001List).stream().collect(Collectors.groupingBy(s->s.getMc001()));
        rstList.forEach(i->{
            List<Invmc> tempList = invmcMap.getOrDefault(i.getMb001(), new ArrayList<Invmc>());
            i.setInvmcs(invmbMapper.toInvmcDto(tempList));
        });
        return rstList;
    }


    public List<InvmaDto> getInvmaList(InvmaCriteria criteria) {
        Specification<Invma> query = new Specification<Invma>() {
            @Override
            public Predicate toPredicate(Root<Invma> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) {
                List<Predicate> predicates = new ArrayList<Predicate>();
                return criteriaQuery.where(predicates.toArray(new Predicate[0])).getRestriction();
            }
        };
        Page<Invma> list = invmaDao.findAll(query, criteria.toPageable());
        List<InvmaDto> rstList = invmbMapper.toInvmaDtoList(list.toList());
        return rstList;
    }


    public List<InvmlDto> getInvmlList(InvmlCriteria criteria) {
        Specification<Invml> query = new Specification<Invml>() {
            @Override
            public Predicate toPredicate(Root<Invml> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) {
                List<Predicate> predicates = new ArrayList<Predicate>();
                if (StringUtils.isNotBlank(criteria.getMl001())) {
                    Predicate predicate = criteriaBuilder.equal(root.get("ml001"), criteria.getMl001() );
                    predicates.add(predicate);
                }
                if (StringUtils.isNotBlank(criteria.getMl002())) {
                    Predicate predicate = criteriaBuilder.equal(root.get("ml002"), criteria.getMl002());
                    predicates.add(predicate);
                }
                return criteriaQuery.where(predicates.toArray(new Predicate[0])).getRestriction();
            }
        };
        List<Invml> list = invmlDao.findAll(query);
        return invmbMapper.toInvmlDto(list);
    }

    /**
     * 物料MES更新标识
     * @param updateParams
     */
    @Transactional
    public void syncMark(@RequestBody InvmbUpdateParams updateParams){
        Specification<Invmb> query = new Specification<Invmb>() {
            @Override
            public Predicate toPredicate(Root<Invmb> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) {
                List<Predicate> predicates = new ArrayList<Predicate>();
                predicates.add(criteriaBuilder.equal(root.get("mb001"), updateParams.getMb001()));
                return criteriaQuery.where(predicates.toArray(new Predicate[0])).getRestriction();
            }
        };
        List<Invmb> list = invmbDao.findAll(query);
        list.forEach(s->{
            s.setUdf08(updateParams.getUdf08());
            invmbDao.save(s);
        });
    }
}
