package com.topsunit.scanservice.ximai.service;

import com.topsunit.scanservice.ximai.common.AppConfig;
import com.topsunit.scanservice.ximai.common.FileView;
import com.topsunit.scanservice.ximai.common.SmbFileSystem;
import com.topsunit.scanservice.ximai.dao.InvmbDao;
import com.topsunit.scanservice.ximai.dao.MoctaDao;
import com.topsunit.scanservice.ximai.dao.MoctgDao;
import com.topsunit.scanservice.ximai.dao.MoctbDao;
import com.topsunit.scanservice.ximai.dto.*;
import com.topsunit.scanservice.ximai.dto.mapper.MoctaMapper;
import com.topsunit.scanservice.ximai.entity.Invml;
import com.topsunit.scanservice.ximai.entity.Mocta;
import com.topsunit.scanservice.ximai.entity.Moctb;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.RequestBody;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * <p>Title: MoctaService</p>
 * <p>Description: 工单单头</p>
 *
 * @author xi.feng
 * @version V1.0
 * @date 2021/11/16
 */
@Service
public class MoctaService {

    @Autowired
    private MoctbDao moctbDao;
    @Autowired
    private AppConfig appConfig;
    @Autowired
    private MoctgDao moctgDao;

    private final MoctaDao moctaDao;

    private final InvmbDao invmbDao;
    private final MoctaMapper moctaMapper;

    public MoctaService(MoctaDao moctaDao, InvmbDao invmbDao, MoctaMapper moctaMapper) {
        this.moctaDao = moctaDao;
        this.invmbDao = invmbDao;
        this.moctaMapper = moctaMapper;
    }

    public Optional<MoctaDocDto> getForDoc(MoctaIdCriteria criteria) {
        return moctaDao.findById(moctaMapper.toMoctaId(criteria))
                .map(moctaMapper::toMoctaDocDto)
                .map(i -> {
                    invmbDao.findById(i.getTa006()).ifPresent(j -> i.setMb029(j.getMb029()));
                    return i;
                });
    }

    public Optional<MoctaDto> get(MoctaIdCriteria criteria) {
        return moctaDao.findById(moctaMapper.toMoctaId(criteria))
                .map(i->{
                    MoctaDto dto = moctaMapper.toMoctaDto(i);
                    String batchNo = this.buildFinishedProductBatch(dto.getTa006());
                    dto.setBatchNo(batchNo);
                    return dto;
                });
    }

    public MoctaDocDto getForDoc2(MoctaIdCriteria criteria) {
        MoctaDocDto dto = moctaDao.findById(moctaMapper.toMoctaId(criteria))
                .map(moctaMapper::toMoctaDocDto).orElse(new MoctaDocDto());
        SmbFileSystem fileSystem = new SmbFileSystem();
        fileSystem.setUri(appConfig.getShareUrl());
        if(StringUtils.isNotBlank(dto.getUdf08())){
            List<FileView> list = fileSystem.searchFiles("/", dto.getUdf08()+".*",false);
            list.forEach(s->{
                dto.getFiles().add(s.getName());
            });
        }
        return dto;
    }


    public List<MoctaDto> getMoctaList(MoctaCriteria criteria) {
        Specification<Mocta> query = new Specification<Mocta>() {
            @Override
            public Predicate toPredicate(Root<Mocta> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) {
                List<Predicate> predicates = new ArrayList<Predicate>();
                if (StringUtils.isNotBlank(criteria.getTa002())) {
                    Predicate predicate = criteriaBuilder.equal(root.get("ta002"), criteria.getTa002() );
                    predicates.add(predicate);
                }
                if (StringUtils.isNotBlank(criteria.getTa001())) {
                    Predicate predicate = criteriaBuilder.equal(root.get("ta001"), criteria.getTa001() );
                    predicates.add(predicate);
                }
                if (StringUtils.isNotBlank(criteria.getTa013())) {
                    Predicate predicate = criteriaBuilder.equal(root.get("ta013"), criteria.getTa013() );
                    predicates.add(predicate);
                }
                if (StringUtils.isNotBlank(criteria.getNeUdf07())) {
                    Predicate predicate = criteriaBuilder.notEqual(root.get("udf07"), criteria.getNeUdf07() );
                    predicates.add(predicate);
                }
                if (StringUtils.isNotBlank(criteria.getGtEqCreateDate())) {
                    Predicate predicate = criteriaBuilder.greaterThanOrEqualTo(root.get("createDate"), criteria.getGtEqCreateDate());
                    predicates.add(predicate);
                }
                if (StringUtils.isNotBlank(criteria.getGtEqModiDate())) {
                    Predicate predicate = criteriaBuilder.greaterThanOrEqualTo(root.get("modiDate"), criteria.getGtEqModiDate());
                    predicates.add(predicate);
                }
                return criteriaQuery.where(predicates.toArray(new Predicate[0])).getRestriction();
            }
        };
        Pageable pageable = criteria.toPageable();
        Page<Mocta> list = moctaDao.findAll(query, pageable);
        List<MoctaDto> rstList =  moctaMapper.toMoctaDtoList(list.toList());
        List<String> ta002List = rstList.stream().map(s->s.getTa002()).collect(Collectors.toList());
        Map<String,List<Moctb>> moctbMap = moctbDao.findByTb002InOrderByTb003AscTb006Asc(ta002List).stream().collect(Collectors.groupingBy(s->s.getTb001()+s.getTb002()));
        rstList.forEach(i->{
            List<Moctb> tempList = moctbMap.getOrDefault(i.getTa001() + i.getTa002(), new ArrayList<>());
            i.setMoctbDtos(moctaMapper.toMoctbDto(tempList));
        });
        return rstList;
    }

    public List<MoctbDto> getMoctbList(MoctbCriteria criteria) {
        Specification<Moctb> query = new Specification<Moctb>() {
            @Override
            public Predicate toPredicate(Root<Moctb> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) {
                List<Predicate> predicates = new ArrayList<Predicate>();
                if (StringUtils.isNotBlank(criteria.getTb001())) {
                    Predicate predicate = criteriaBuilder.equal(root.get("tb001"), criteria.getTb001() );
                    predicates.add(predicate);
                }
                if (StringUtils.isNotBlank(criteria.getTb002())) {
                    Predicate predicate = criteriaBuilder.equal(root.get("tb002"), criteria.getTb002() );
                    predicates.add(predicate);
                }
                return criteriaQuery.where(predicates.toArray(new Predicate[0])).getRestriction();
            }
        };
        List<Moctb> list = moctbDao.findAll(query);
        return moctaMapper.toMoctbDto(list);
    }

    @Transactional
    public void updateSyncMark(@RequestBody MoctaUpdateParams params){
        Specification<Mocta> query = new Specification<Mocta>() {
            @Override
            public Predicate toPredicate(Root<Mocta> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) {
                List<Predicate> predicates = new ArrayList<Predicate>();
                predicates.add(criteriaBuilder.equal(root.get("ta002"), params.getTa002() ));
                predicates.add(criteriaBuilder.equal(root.get("ta001"), params.getTa001() ));
                return criteriaQuery.where(predicates.toArray(new Predicate[0])).getRestriction();
            }
        };
        List<Mocta> list = moctaDao.findAll(query);
        list.forEach(s->{
            s.setUdf07(params.getUdf07());
            moctaDao.save(s);
        });
    }

    @Transactional
    public void updateState(@RequestBody MoctaUpdateParams params){
        Specification<Mocta> query = new Specification<Mocta>() {
            @Override
            public Predicate toPredicate(Root<Mocta> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) {
                List<Predicate> predicates = new ArrayList<Predicate>();
                predicates.add(criteriaBuilder.equal(root.get("ta002"), params.getTa002() ));
                predicates.add(criteriaBuilder.equal(root.get("ta001"), params.getTa001() ));
                return criteriaQuery.where(predicates.toArray(new Predicate[0])).getRestriction();
            }
        };
        List<Mocta> list = moctaDao.findAll(query);
        list.forEach(s->{
            s.setTa011(params.getTa011());
            moctaDao.save(s);
        });
    }

    public String buildFinishedProductBatch(String materialNo) {
        String dateStr = cn.hutool.core.date.DateUtil.format(java.time.LocalDateTime.now(), "yyyyMMdd");
        String prefix = (materialNo == null ? "" : materialNo.trim()) + dateStr;
        com.topsunit.scanservice.ximai.entity.Moctg last = moctgDao.findFirstByTg017StartingWithOrderByTg017Desc(prefix);
        int nextSeq = 1;
        if (last != null && last.getTg017() != null) {
            String lastBatch = last.getTg017().trim();
            if (lastBatch.length() >= 3) {
                String tail = lastBatch.substring(lastBatch.length() - 3);
                try {
                    nextSeq = Integer.parseInt(tail) + 1;
                } catch (NumberFormatException ignore) {
                    nextSeq = 1;
                }
            }
        }
        String seqStr = String.format("%03d", nextSeq);
        return prefix + seqStr;
    }
}
