package com.topsunit.scanservice.ximai.common;

import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * 文件系统
 * <p/>
 * 该接口定义一个与具体实现无关的文件系统接口
 *
 * @author vacoor
 */
public abstract class FileSystem {

    /**
     * 给定路径是否存在
     *
     * @param path 文件系统路径
     * @throws FileSystemException
     */
    public abstract boolean exists(String path) throws FileSystemException;

    /**
     * 获取路径的文件状态
     *
     * @param path 文件系统路径
     * @return 返回文件状态, 如果文件不存在返回 null
     * @throws FileSystemException
     */
    public abstract FileView stat(String path) throws FileSystemException;

    /**
     * 获取路径的文件状态
     *
     * @param path 文件或目录路径
     * @return 文件或目录的状态, 如果目录下没有文件或文件不存在返回空数组
     * @throws FileSystemException
     */
    public abstract FileView[] ls(String path) throws FileSystemException;

    /**
     * 创建给定路径的目录, 如果目录已经创建成功或已经存在返回 true, 创建失败返回 false
     * 如果已经存在但不是一个目录则抛出异常
     *
     * @param path 目录路径
     * @throws FileSystemException
     */
    public abstract boolean mkdirs(String path) throws FileSystemException;

    /**
     * 打开给定路径的文件, 如果给定路径不存在返回null, 如果不是文件将抛出异常
     * <p/>
     * 注: 关闭输入流时才会关闭相关资源, 因此读取完毕后务必关闭输入流
     *
     * @param file 文件路径
     * @return 文件输入流或null
     * @throws FileSystemException
     */
    public abstract InputStream open(String file) throws FileSystemException;

    /**
     * 获取一个创建文件的输出流并自动创建相关目录, 当 override = false, 如果文件已经存在则抛出异常
     *
     * @param file     要创建的文件路径
     * @param override 如果文件已经存在是否覆盖
     * @return 新建文件的输出流
     * @throws FileSystemException
     */
    public abstract OutputStream create(String file, boolean override) throws FileSystemException;

    /**
     * 获取一个创建文件的输出流并自动创建相关目录, 当 override = false, 如果文件已经存在则抛出异常
     *
     * @param file        要创建的文件路径
     * @param contentType MIME 类型, 某些文件系统会记录该值, 例如 OSS 在返回时使用
     * @param override    如果文件已经存在是否覆盖
     * @return 新建文件的输出流
     * @throws FileSystemException
     */
    public abstract OutputStream create(String file, String contentType, boolean override) throws FileSystemException;

    /**
     * 删除给定路径
     *
     * @param path      文件或目录路径
     * @param force     如果文件夹中存在文件是否强制删除
     * @param recursive 如果文件夹中存在目录是否递归删除
     * @throws FileSystemException
     */
    public abstract void rm(String path, boolean force, boolean recursive) throws FileSystemException;

    /**
     * 重命名路径
     *
     * @param oldPath 源路径
     * @param newPath 目标路径
     * @throws FileSystemException
     */
    public abstract void rename(String oldPath, String newPath) throws FileSystemException;

    /**
     * 关闭连接通道
     * @throws FileSystemException
     */
    public abstract void disconnect() throws FileSystemException;

    /**
     * 搜索指定名称文件
     * @param path 路径
     * @param name 支持正则
     * @param recursive 是否搜索所有层级目录，false：只在当前目录查找
     */
    public List<FileView> searchFiles(String path, String name, boolean recursive) {
        List<FileView> rst = new ArrayList<FileView>();
        FileView[] fileViews = this.ls(path);
        for(FileView fileView : fileViews){
            if(fileView.isDirectory()){
                if(recursive)
                    rst.addAll(this.searchFiles(fileView.getPath(), name, recursive));
            }else if(fileView.getName().matches(name)){
                rst.add(fileView);
            }
        }
        return rst;
    }

    /**
     * 搜索指定名称文件
     * @param path 路径
     * @param matchFile 文件区配接口
     * @param recursive 是否搜索所有层级目录，false：只在当前目录查找
     */
    public List<FileView> searchFiles(String path, MatchFile matchFile, boolean recursive) {
        List<FileView> rst = new ArrayList<FileView>();
        FileView[] fileViews = this.ls(path);
        for(FileView fileView : fileViews){
            if(fileView.isDirectory()){
                if(recursive)
                    rst.addAll(this.searchFiles(fileView.getPath(), matchFile, recursive));
            }else if(matchFile.isMatch(fileView.getName(), fileView.getPath())){
                rst.add(fileView);
            }
        }
        return rst;
    }


    public void zip(String archive, String... files) throws IOException {
        File archivePath = new File(archive);
        File parent = archivePath.getParentFile();
        if (!parent.exists() && !parent.mkdirs()) {
            throw new IOException("cannot mkdirs: " + parent);
        }
        zip(new FileOutputStream(archivePath), files);
    }

    public void zip(OutputStream out, String... files) throws IOException {
        Path[] paths = new Path[files.length];
        for (int i = 0; i < files.length; i++) {
            paths[i] = Path.get(files[i]);
        }

        compressTo(out, paths);
    }

    public void compressTo(OutputStream out, Path[] files) throws IOException {
        // ZipOutputStream zipOut = new ZipOutputStream(out, Charsets.UTF_8);
        ZipOutputStream zipOut = new ZipOutputStream(out);
        try {
            for (Path f : files) {
                doCompressEntry(zipOut, f, f.getParent());
            }
        } finally {
            zipOut.flush();
            zipOut.close();
        }
    }

    private void doCompressEntry(ZipOutputStream zipOut, Path file, Path baseDir) throws IOException {
        FileView view = this.stat(file.getPath());
        if (null == view) {
            // not exists
            return;
        }


        String relativePath = file.relativize(baseDir).getPath();
        // 如果要添加一个目录,注意使用符号"/"作为添加项名字结尾符
        relativePath = view.isDirectory() && !relativePath.endsWith("/") ? relativePath + "/" : relativePath;
        relativePath = relativePath.startsWith("/") || relativePath.startsWith("\\") ? relativePath.substring(1) : relativePath;

        // 添加一个 entry
        zipOut.putNextEntry(new ZipEntry(relativePath));
        if (!view.isDirectory()) { // 如果是文件, 写入文件内容
            InputStream in = this.open(file.getPath());
            if (null != in) {
                IOUtils.flow(in, zipOut, true, false);
            }
        }
        zipOut.closeEntry();

        // 如果是真实目录(非符号链接)遍历其下所有文件
        if (view.isDirectory() && !view.isSymlink()) {
            FileView[] views = this.ls(file.getPath());
            for (FileView f : views) {
                String path = f.getPath();
                path = !path.startsWith("/") ? "/" + path : path;
                doCompressEntry(zipOut, Path.get(path), baseDir);
            }
        }
    }

}
