package com.topsunit.scanservice.ximai.barcode.service;

import com.topsunit.scanservice.ximai.barcode.dao.ATCstDsDao;
import com.topsunit.scanservice.ximai.barcode.dao.ATCstDmDao;
import com.topsunit.scanservice.ximai.barcode.dto.ATCstDsDto;
import com.topsunit.scanservice.ximai.barcode.dto.BoxCodeQueryResultDto;
import com.topsunit.scanservice.ximai.barcode.entity.ATCstDs;
import com.topsunit.scanservice.ximai.barcode.entity.ATCstDm;
import com.topsunit.scanservice.ximai.common.TopsunitException;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * <p>Title: ATCstDsService</p>
 * <p>Description: 派工明细服务类</p>
 *
 * @author system
 * @version V1.0
 * @date 2025/01/23
 */
@Service
public class ATCstDsService {

    @Autowired
    private ATCstDsDao atCstDsDao;
    
    @Autowired
    private ATCstDmDao atCstDmDao;

    /**
     * 根据ID获取派工明细信息
     */
    public ATCstDsDto getById(String id) {
        Optional<ATCstDs> optional = atCstDsDao.findById(id);
        if (optional.isPresent()) {
            ATCstDsDto dto = new ATCstDsDto();
            BeanUtils.copyProperties(optional.get(), dto);
            return dto;
        }
        throw new TopsunitException("未找到指定记录");
    }

    /**
     * 获取派工明细列表
     */
    public Page<ATCstDsDto> getList(ATCstDsDto criteria, int page, int size) {
        Pageable pageable = PageRequest.of(page, size);
        
        Specification<ATCstDs> spec = new Specification<ATCstDs>() {
            @Override
            public Predicate toPredicate(Root<ATCstDs> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
                List<Predicate> predicates = new ArrayList<>();
                
                if (criteria.getSItemNo() != null && !criteria.getSItemNo().isEmpty()) {
                    predicates.add(cb.like(root.get("sItemNo"), "%" + criteria.getSItemNo() + "%"));
                }
                if (criteria.getSItemNo_1() != null && !criteria.getSItemNo_1().isEmpty()) {
                    predicates.add(cb.like(root.get("sItemNo_1"), "%" + criteria.getSItemNo_1() + "%"));
                }
                if (criteria.getSItemName() != null && !criteria.getSItemName().isEmpty()) {
                    predicates.add(cb.like(root.get("sItemName"), "%" + criteria.getSItemName() + "%"));
                }
                if (criteria.getSpec() != null && !criteria.getSpec().isEmpty()) {
                    predicates.add(cb.like(root.get("spec"), "%" + criteria.getSpec() + "%"));
                }
                if (criteria.getSnNo() != null && !criteria.getSnNo().isEmpty()) {
                    predicates.add(cb.like(root.get("snNo"), "%" + criteria.getSnNo() + "%"));
                }
                if (criteria.getSnno1() != null && !criteria.getSnno1().isEmpty()) {
                    predicates.add(cb.like(root.get("snno1"), "%" + criteria.getSnno1() + "%"));
                }
                if (criteria.getItemmgpk() != null && !criteria.getItemmgpk().isEmpty()) {
                    predicates.add(cb.equal(root.get("itemmgpk"), criteria.getItemmgpk()));
                }
                if (criteria.getSgpk() != null && !criteria.getSgpk().isEmpty()) {
                    predicates.add(cb.equal(root.get("sgpk"), criteria.getSgpk()));
                }
                
                return cb.and(predicates.toArray(new Predicate[0]));
            }
        };
        
        Page<ATCstDs> pageResult = atCstDsDao.findAll(spec, pageable);
        Page<ATCstDsDto> dtoPage = pageResult.map(entity -> {
            ATCstDsDto dto = new ATCstDsDto();
            BeanUtils.copyProperties(entity, dto);
            return dto;
        });
        
        return dtoPage;
    }

    /**
     * 保存派工明细信息
     */
    @Transactional
    public ATCstDsDto save(ATCstDsDto dto) {
        try {
            ATCstDs entity = new ATCstDs();
            BeanUtils.copyProperties(dto, entity);
            ATCstDs saved = atCstDsDao.save(entity);
            
            ATCstDsDto result = new ATCstDsDto();
            BeanUtils.copyProperties(saved, result);
            return result;
        } catch (Exception e) {
            throw new TopsunitException("保存失败: " + e.getMessage());
        }
    }

    /**
     * 更新派工明细信息
     */
    @Transactional
    public ATCstDsDto update(ATCstDsDto dto) {
        try {
            if (dto.getId() == null || dto.getId().isEmpty()) {
                throw new TopsunitException("ID不能为空");
            }
            
            Optional<ATCstDs> optional = atCstDsDao.findById(dto.getId());
            if (!optional.isPresent()) {
                throw new TopsunitException("未找到指定记录");
            }
            
            ATCstDs entity = optional.get();
            BeanUtils.copyProperties(dto, entity);
            ATCstDs saved = atCstDsDao.save(entity);
            
            ATCstDsDto result = new ATCstDsDto();
            BeanUtils.copyProperties(saved, result);
            return result;
        } catch (Exception e) {
            throw new TopsunitException("更新失败: " + e.getMessage());
        }
    }

    /**
     * 删除派工明细信息
     */
    @Transactional
    public String delete(String id) {
        try {
            if (id == null || id.isEmpty()) {
                throw new TopsunitException("ID不能为空");
            }
            
            if (!atCstDsDao.existsById(id)) {
                throw new TopsunitException("未找到指定记录");
            }
            
            atCstDsDao.deleteById(id);
            return "删除成功";
        } catch (Exception e) {
            throw new TopsunitException("删除失败: " + e.getMessage());
        }
    }

    /**
     * 通过箱码查询小标签条码
     */
    public List<BoxCodeQueryResultDto> getSmallLabelBarcodesByBoxCode(String boxCode) {
        try {
            if (boxCode == null || boxCode.isEmpty()) {
                throw new TopsunitException("箱码不能为空");
            }
            
            // 先通过箱码查询ATCstDm表获取mgpk
            List<ATCstDm> atCstDmList = atCstDmDao.findByPSNNo(boxCode);
            if (atCstDmList.isEmpty()) {
                throw new TopsunitException("未找到对应的箱码记录");
            }
            
            List<BoxCodeQueryResultDto> resultList = new ArrayList<>();
            
            for (ATCstDm atCstDm : atCstDmList) {
                // 通过mgpk查询ATCstDs表获取小标签条码
                List<ATCstDs> atCstDsList = atCstDsDao.findByMgpk(atCstDm.getMgpk());
                
                for (ATCstDs atCstDs : atCstDsList) {
                    BoxCodeQueryResultDto result = new BoxCodeQueryResultDto();
                    
                    // 复制ATCstDm字段
                    result.setEmplID(atCstDm.getEmplid());
                    result.setEmplNo(atCstDm.getEmplno());
                    result.setEmplName(atCstDm.getEmplname());
                    result.setDeptId(atCstDm.getDeptid());
                    result.setDeptNo(atCstDm.getDeptno());
                    result.setDeptName(atCstDm.getDeptname());
                    result.setMNo(atCstDm.getMno());
                    result.setBdt(atCstDm.getBdt());
                    result.setMItemID(atCstDm.getMitemid());
                    result.setMitemno(atCstDm.getMitemno());
                    result.setMitemno_1(atCstDm.getMitemno_1());
                    result.setMItemName(atCstDm.getMitemname());
                    result.setYmdt(atCstDm.getYmdt());
                    result.setMQty(atCstDm.getMqty());
                    result.setPSNNo(atCstDm.getPsnno());
                    result.setTrick(atCstDm.getTrick());
                    result.setMNote(atCstDm.getMnote());
                    result.setCuser(atCstDm.getCuser());
                    result.setCdt(atCstDm.getCdt());
                    result.setMgpk(atCstDm.getMgpk());
                    
                    // 复制ATCstDs字段
                    result.setItemmgpk(atCstDs.getItemmgpk());
                    result.setSItemNo(atCstDs.getSItemNo());
                    result.setSItemNo_1(atCstDs.getSItemNo_1());
                    result.setSItemName(atCstDs.getSItemName());
                    result.setSpec(atCstDs.getSpec());
                    result.setItemrpack(atCstDs.getItemrpack());
                    result.setSnqty(atCstDs.getSnqty());
                    result.setSnNo(atCstDs.getSnNo());
                    result.setSnno1(atCstDs.getSnno1());
                    result.setSrNote(atCstDs.getSrNote());
                    result.setSurid(atCstDs.getSurid());
                    result.setSauid(atCstDs.getSauid());
                    result.setSgpk(atCstDs.getSgpk());
                    
                    resultList.add(result);
                }
            }
            
            return resultList;
        } catch (Exception e) {
            throw new TopsunitException("查询失败: " + e.getMessage(), e);
        }
    }
}
