package com.topsunit.scanservice.ximai.barcode.service;

import com.topsunit.scanservice.ximai.barcode.dao.ATCstDmDao;
import com.topsunit.scanservice.ximai.barcode.dto.ATCstDmDto;
import com.topsunit.scanservice.ximai.barcode.entity.ATCstDm;
import com.topsunit.scanservice.ximai.common.TopsunitException;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * <p>Title: ATCstDmService</p>
 * <p>Description: 在线标签打印服务类</p>
 *
 * @author system
 * @version V1.0
 * @date 2025/01/23
 */
@Service
public class ATCstDmService {

    @Autowired
    private ATCstDmDao atCstDmDao;

    /**
     * 根据ID获取在线标签打印信息
     */
    public ATCstDmDto getById(String id) {
        Optional<ATCstDm> optional = atCstDmDao.findById(id);
        if (optional.isPresent()) {
            ATCstDmDto dto = new ATCstDmDto();
            BeanUtils.copyProperties(optional.get(), dto);
            return dto;
        }
        throw new TopsunitException("未找到指定记录");
    }

    /**
     * 获取在线标签打印列表
     */
    public Page<ATCstDmDto> getList(ATCstDmDto criteria, int page, int size) {
        Pageable pageable = PageRequest.of(page, size);
        
        Specification<ATCstDm> spec = new Specification<ATCstDm>() {
            @Override
            public Predicate toPredicate(Root<ATCstDm> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
                List<Predicate> predicates = new ArrayList<>();
                
                if (criteria.getMNo() != null && !criteria.getMNo().isEmpty()) {
                    predicates.add(cb.like(root.get("mNo"), "%" + criteria.getMNo() + "%"));
                }
                if (criteria.getMitemno() != null && !criteria.getMitemno().isEmpty()) {
                    predicates.add(cb.like(root.get("mitemno"), "%" + criteria.getMitemno() + "%"));
                }
                if (criteria.getMItemName() != null && !criteria.getMItemName().isEmpty()) {
                    predicates.add(cb.like(root.get("mItemName"), "%" + criteria.getMItemName() + "%"));
                }
                if (criteria.getBdt() != null && !criteria.getBdt().isEmpty()) {
                    predicates.add(cb.equal(root.get("bdt"), criteria.getBdt()));
                }
                if (criteria.getYmdt() != null && !criteria.getYmdt().isEmpty()) {
                    predicates.add(cb.equal(root.get("ymdt"), criteria.getYmdt()));
                }
                
                return cb.and(predicates.toArray(new Predicate[0]));
            }
        };
        
        Page<ATCstDm> pageResult = atCstDmDao.findAll(spec, pageable);
        Page<ATCstDmDto> dtoPage = pageResult.map(entity -> {
            ATCstDmDto dto = new ATCstDmDto();
            BeanUtils.copyProperties(entity, dto);
            return dto;
        });
        
        return dtoPage;
    }

    /**
     * 保存在线标签打印信息
     */
    @Transactional
    public ATCstDmDto save(ATCstDmDto dto) {
        try {
            ATCstDm entity = new ATCstDm();
            BeanUtils.copyProperties(dto, entity);
            ATCstDm saved = atCstDmDao.save(entity);
            
            ATCstDmDto result = new ATCstDmDto();
            BeanUtils.copyProperties(saved, result);
            return result;
        } catch (Exception e) {
            throw new TopsunitException("保存失败: " + e.getMessage());
        }
    }

    /**
     * 更新在线标签打印信息
     */
    @Transactional
    public ATCstDmDto update(ATCstDmDto dto) {
        try {
            if (dto.getId() == null || dto.getId().isEmpty()) {
                throw new TopsunitException("ID不能为空");
            }
            
            Optional<ATCstDm> optional = atCstDmDao.findById(dto.getId());
            if (!optional.isPresent()) {
                throw new TopsunitException("未找到指定记录");
            }
            
            ATCstDm entity = optional.get();
            BeanUtils.copyProperties(dto, entity);
            ATCstDm saved = atCstDmDao.save(entity);
            
            ATCstDmDto result = new ATCstDmDto();
            BeanUtils.copyProperties(saved, result);
            return result;
        } catch (Exception e) {
            throw new TopsunitException("更新失败: " + e.getMessage());
        }
    }

    /**
     * 删除在线标签打印信息
     */
    @Transactional
    public String delete(String id) {
        try {
            if (id == null || id.isEmpty()) {
                throw new TopsunitException("ID不能为空");
            }
            
            if (!atCstDmDao.existsById(id)) {
                throw new TopsunitException("未找到指定记录");
            }
            
            atCstDmDao.deleteById(id);
            return "删除成功";
        } catch (Exception e) {
            throw new TopsunitException("删除失败: " + e.getMessage());
        }
    }
}
