package com.huigou.uasp.bmp.fn.impl;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.huigou.context.Operator;
import com.huigou.context.ThreadLocalUtil;
import com.huigou.data.domain.query.QueryPageRequest;
import com.huigou.uasp.bmp.configuration.domain.model.CodeRecycle;
import com.huigou.uasp.bmp.configuration.domain.query.CodeRecycleQuery;
import com.huigou.uasp.bmp.configuration.mapper.CodeRecycleMapper;
import com.huigou.uasp.bmp.configuration.repository.CodeBuildRuleRepository;
import com.huigou.util.ClassHelper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import com.huigou.exception.ApplicationException;
import com.huigou.uasp.bmp.configuration.application.CodeBuildRuleApplication;
import com.huigou.uasp.bmp.configuration.domain.model.CodeBuildRule;
import com.huigou.uasp.bmp.fn.AbstractDaoFunction;
import com.huigou.uasp.bmp.fn.CodeGenerator;
import com.huigou.util.DateUtil;

/**
 * 单据编码规则生成器
 * 
 * @author xx
 */
@Service("codeGenerator")
public class CodeGeneratorImpl extends AbstractDaoFunction implements CodeGenerator {

    @Autowired
    private CodeBuildRuleApplication codeBuildRuleApplication;
    @Autowired
    private CodeBuildRuleRepository codeBuildRuleRepository;
    @Autowired
    private CodeRecycleMapper codeRecycleMapper;

    /**
     * 获取单据编号
     * 
     * @return
     */
    public String getNextCode(String bizKindId) {
        StringBuffer sb = new StringBuffer();
        try {
            CodeBuildRule codeBuildRule = codeBuildRuleApplication.getRuleValueAsStep(bizKindId, 1);
            //查询编号回收表是否存在值，存在的话直接返回
            List<String> noList = this.useRecycleCode(codeBuildRule.getCode(), 1);
            if(noList.size()>0){
                return noList.get(0);
            }
            String rule = codeBuildRule.getRule();
            Integer nextValue = codeBuildRule.getCurrentValue();
            Pattern p = Pattern.compile(CodeBuildRuleApplication.RULE_REG, Pattern.CASE_INSENSITIVE);
            Matcher m = p.matcher(rule);
            if (m.find()) {
                String dateFormat = m.group(1);
                int serialLength = Integer.parseInt(m.group(3));
                m.appendReplacement(sb, formatDate(dateFormat, DateUtil.getDateTime()) + m.group(2) + formatSerialNumber(nextValue, serialLength));
            }
        } catch (Exception e) {
            throw new ApplicationException(e);
        }
        return sb.toString();
    }

    /**
     * 按步长批量取号
     * 
     * @param bizKindId
     * @param step
     */
    public List<String> getNextCodesAsStep(String bizKindId, Integer step) {
        // 步长最小为1
        if (step == null || step < 1) {
            step = 1;
        }
        try {
            //先查询编号回收表是否存在值，存在的话优先取
            CodeBuildRule tempBuildRule = codeBuildRuleRepository.getFirstByCode(bizKindId);
            List<String> noList = this.useRecycleCode(tempBuildRule.getCode(), step);
            //回收表编号不足的情况，继续生成编号
            if(step > noList.size()){
                step = step - noList.size();
                CodeBuildRule codeBuildRule = codeBuildRuleApplication.getRuleValueAsStep(bizKindId, step);
                String rule = codeBuildRule.getRule();
                Integer lastValue = codeBuildRule.getLastValue();
                Pattern p = Pattern.compile(CodeBuildRuleApplication.RULE_REG, Pattern.CASE_INSENSITIVE);
                int serialLength = 1;
                StringBuffer sb = new StringBuffer();
                Matcher m = p.matcher(rule);
                if (m.find()) {
                    String dateFormat = m.group(1);
                    serialLength = Integer.parseInt(m.group(3));
                    m.appendReplacement(sb, formatDate(dateFormat, DateUtil.getDateTime()) + m.group(2) + "%s");
                }
                for (int i = 1; i <= step; i++) {
                    noList.add(String.format(sb.toString(), formatSerialNumber(lastValue + i, serialLength)));
                }
            }
            return noList;
        } catch (Exception e) {
            throw new ApplicationException(e);
        }
    }

    @Override
    public void recycleCodeNos(String code,String[] codesNos) {
        //保存回收编号数据
        for(String temp : codesNos){
            CodeRecycle codeRecycle = new CodeRecycle();
            codeRecycle.setCode(code);
            codeRecycle.setCodeNo(temp);
            codeRecycle.setIsRecycle("0");
            Operator operator = ThreadLocalUtil.getOperator();
            codeRecycle.setCreatedDate(new Date());
            codeRecycle.setCreatedById(operator.getLoginUser().getId());
            codeRecycle.setCreatedByName(operator.getLoginUser().getName());
            codeBuildRuleApplication.saveCodeRecycle(codeRecycle);
        }
    }

    /**
     * 格式化序列号
     * 
     * @param serialNumber
     *            序列号
     * @param length
     *            长度
     * @return
     */
    private String formatSerialNumber(Integer serialNumber, int length) {
        String result = String.valueOf(serialNumber);
        Assert.isTrue(result.length() <= length, String.format("流水号长度超出限制，设置长度：%s，实际长度：%s。", length, result.length()));
        while (result.length() < length) {
            result = "0" + result;
        }
        return result;
    }

    /**
     * 更具单据编号规则格式化日期
     * 
     * @param dateFormat
     *            日期格式
     * @param date
     *            日期
     * @return
     */
    private String formatDate(String dateFormat, Date date) {
        SimpleDateFormat df = new SimpleDateFormat(dateFormat);
        return df.format(date);
    }

    /*
    使用回收自定义编号值
     */
    private List<String> useRecycleCode(String code, int quantity) {
        CodeRecycleQuery query = new CodeRecycleQuery();
        query.setCode(code);
        query.setIsRecycle("0");
        QueryPageRequest pageInfo = new QueryPageRequest();
        pageInfo.setPageIndex(1);
        pageInfo.setPageSize(quantity);
        pageInfo.setSortFieldName("CREATED_DATE");
        query.setPageModel(pageInfo);
        Map<String, Object> data = codeRecycleMapper.slicedCodeRecycle(query);
        List<CodeRecycle> list = ((List<Object>)data.get("Rows")).stream().map(map->ClassHelper.fromMap(CodeRecycle.class,(Map<String,Object>)map)).collect(Collectors.toList());
        //返回回收的自定义编码值
        List<String> rst = new ArrayList<String>();
        for(int i=0;i<quantity&&i<list.size();i++){
            CodeRecycle codeRecycle = list.get(i);
            rst.add(codeRecycle.getCodeNo());
            //修改回收数据为已使用
            codeRecycle.setUseDate(new Date());
            codeRecycle.setIsRecycle("1");
            codeBuildRuleApplication.saveCodeRecycle(codeRecycle);
        }
        return rst;
    }
}
