package com.huigou.uasp.bmp.doc.attachment.application.impl;

import com.huigou.cache.DictionaryDesc;
import com.huigou.cache.SystemCache;
import com.huigou.uasp.bmp.doc.attachment.application.AttachmentSecretInfoResolver;
import com.huigou.uasp.bmp.doc.attachment.domain.model.FileInfo;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.Assert;

import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 这是一个测试类，该类认为附件名的格式为【密级名称】xxx.xx,如 【非密】报销单.pdf
 *
 * @author yonghuan
 */
public class TestAttachmentSecretInfoResolver extends AbstractAttachmentSecretInfoResolver implements AttachmentSecretInfoResolver {
    /**
     * 附件文件名格式,如 【非密】报销单.pdf
     */
    private final static Pattern FILE_NAME_PATTERN = Pattern.compile("^【(.+)】.+$");
    /**
     * 密级字典编码
     */
    private String securityGradeDictionaryCode;

    @Value("${securityGradeDictionaryCode}")
    public void setSecurityGradeDictionaryCode(String securityGradeDictionaryCode) {
        this.securityGradeDictionaryCode = securityGradeDictionaryCode;
    }

    @Override
    protected String resolveAttachmentSecurityLevel(FileInfo fileInfo) {
        Matcher matcher = FILE_NAME_PATTERN.matcher(fileInfo.getName());
        Assert.isTrue(matcher.matches(), "附件名不合法，合法的附件名为：【密级】文件名，如：【非密】报销单.pdf");
        String attachmentSecurityGradeName = matcher.group(1);
        Assert.hasText(attachmentSecurityGradeName, "附件名中未包含附件密级信息");
        DictionaryDesc attachmentSecurityGrade = SystemCache.getDictionary(securityGradeDictionaryCode).values()
                .stream()
                .filter(e -> Objects.equals(e.getName(), attachmentSecurityGradeName))
                .findAny()
                .orElseThrow(() -> new IllegalArgumentException(String.format("无效的附件密级：%s", attachmentSecurityGradeName)));
        return attachmentSecurityGrade.getValue();
    }
}
