import axios from 'axios'
import Qs from 'qs'
import { Toast } from 'vant'
import store from '@/store'
import router from '@/router'
import { getToken } from '@/utils/auth'

// create an axios instance
const service = axios.create({
  baseURL: '',
  // withCredentials: true, // send cookies when cross-domain requests
  timeout: 1000 * 30, // request timeout
  headers: {
    'Cache-Control': 'no-cache',
    'Content-Type': 'application/json;charset=utf-8'
  }
})

// request interceptor
service.interceptors.request.use(
  config => {
    // do something before request is sent

    if (
      (config.url.indexOf('login') > 0 || config.headers['needFormat']) &&
      config.data &&
      config.headers['Content-Type'] === 'application/x-www-form-urlencoded'
    ) {
      config.data = Qs.stringify(config.data, {
        arrayFormat: 'repeat'
      })
    }

    if (store.getters.token) {
      config.headers['X-Auth-Token'] = getToken()
    }
    return config
  },
  error => {
    // do something with request error
    console.log(error) // for debug
    return Promise.reject(error)
  }
)

// response interceptor
service.interceptors.response.use(
  /**
   * If you want to get http information such as headers or status
   * Please return  response => response
   */

  /**
   * Determine the request status by custom code
   * Here is just an example
   * You can also judge the status by HTTP Status Code
   */
  async response => {
    // console.log('22', response, typeof response.data)
    // stream形式
    if (
      response.data instanceof Blob ||
      response.headers['content-type'].indexOf('octet-stream') !== -1
    ) {
      return {
        data: response.data,
        type: response.headers['content-type'],
        name: decodeURIComponent(response.headers['content-disposition'].split('=')[1]).replaceAll(
          '"',
          ''
        )
      }
    }
    const dataAxios = response.data
    const { status, message, msg } = dataAxios
    // scm接口
    if (response.config.url.indexOf('scm') !== -1) {
      return dataAxios
    }
    if (response.config.responseType === 'arraybuffer') {
      // 刷新tonken
      return response
    }

    if (status === 401) {
      Toast(msg)
      await store.dispatch('resetToken')
      router.replace('/login')
      return
    }
    if (status === '2') {
      Toast(message || msg)
      return dataAxios
    }
    return dataAxios
  },
  async error => {
    // const { data } = error.response
    console.error('error: ', error) // for debug
    if (`${error}`.indexOf('401') !== -1) {
      Toast(`登录用户失效，请重新登录`)
      await store.dispatch('resetToken')
      router.replace('/login')
      return
    }

    return Promise.reject(error)
  }
)

export default service
