import type { Recordable } from '@/types/global'
import { defineStore } from 'pinia'

type StoreState = {
  marks: Record<any, any>
  fileInfo: Record<any, any>
  isSelectedRow: Boolean
  isSetSheetMarks: Boolean[]
  sheetBegin: Record<any, any>[],
}

export const useExcelStore = defineStore('excelStore', {
  state: (): StoreState => ({
    marks: {},
    fileInfo: {},
    isSelectedRow: false,
    isSetSheetMarks: [],
    sheetBegin: []
  }),
  getters: {
    getMarks(): Record<any, any> {
      return this.marks
    },
    getFileInfo(): Record<any, any> {
      return this.fileInfo
    },
    getIsSelectedRow(): Boolean {
      return this.isSelectedRow
    },
    getIsSetSheetMarks(): boolean[] {
      return this.isSetSheetMarks as []
    },
    getSheetBegin(): Record<any, any>[] {
      return this.sheetBegin
    }
  },
  actions: {
    setFileInfo(info: any) {
      Object.assign(this.fileInfo, info)
    },
    deleteMark(order: any, r: any, c: any) {
      const ckey = r + '_' + c
      Reflect.deleteProperty(this.marks[order], ckey)
    },
    setMark(order: any, r: any, c: any, data: any) {
      const ckey = r + '_' + c
      if (!Reflect.has(this.marks, order)) {
        Reflect.set(this.marks, order, {})
      }
      Reflect.set(this.marks[order], ckey, data)
    },
    hasMark(order: any, r: any, c: any) {
      const ckey = r + '_' + c
      if (!Reflect.has(this.marks, order) || !Reflect.has(this.marks[order], ckey)) {
        return false
      }
      return true
    },
    getMark(order: any, r: any, c: any) {
      const ckey = r + '_' + c
      if (!this.hasMark(order, r, c)) {
        return false
      }
      return Reflect.get(this.marks[order], ckey)
    },
    setIsSelectedRow(flag: boolean) {
      this.isSelectedRow = flag
    },
    initSheetBegin(length: number) {
      this.sheetBegin = Array.from({ length }).map((v) => ({})) as []
    },
    setSheetBegin(order: any, data: any) {
      this.sheetBegin[order] = data
    },
    initIsSetSheetMarks(length: number) {
      this.isSetSheetMarks = Array.from({ length }).fill(false) as boolean[]
    },
    setIsSetSheetMark(order: number) {
      this.isSetSheetMarks[order] = true
    },
    resetMarks() {
      this.marks = {}
    },
    resetFileInfo() {
      this.fileInfo = {}
    },
    resetState() {
      this.resetMarks()
    }
  }
})

type ExcelChangeStore = {
  allMapConfigs: Recordable
  scriptList: Recordable[]
  orderFieldList: Recordable[]
  usedFieldList: Recordable[]
  specList?: Recordable[]
  filterMapField?: Recordable[]
}

export const useExcelChangeStore = defineStore('excelChangeStore', {
  state: (): ExcelChangeStore => ({
    allMapConfigs: {}, // 缓存当前这条数据的所有映射关系
    scriptList: [], // 脚本
    orderFieldList: [], // 订单字段
    usedFieldList: [] // 当前编辑column使用过的映射字段
  }),
  getters: {
    getAllMapConfigs(): Recordable {
      return this.allMapConfigs
    },
    getScriptList(): any {
      return this.scriptList
    },
    getEffectConfigs(): any[] {
      // 获取已经使用的config，用于禁用已经选择过的映射字段
      const currenttitle = this.usedFieldList[0]?.title || ''
      const res: any = [...this.usedFieldList]
      for (const key in this.allMapConfigs) {
        if (key != currenttitle) {
          if (this.allMapConfigs[key].orderFiledConfs.length) {
            res.push(...this.allMapConfigs[key].orderFiledConfs.filter((item) => item.variable))
          }
        }
      }
      return res.map((item) => item.mapField)
    },
    getAllFields(): any {
      return this.orderFieldList.map((item) => {
        this.getEffectConfigs.includes(item.filedName)
          ? (item.disabled = true)
          : (item.disabled = false)
        return item
      })
    }
  },
  actions: {
    setAllMapConfigs(config: any) {
      this.allMapConfigs = config
    },
    setSingleFieldMap(config: any) {
      this.allMapConfigs[config.title] = config
    },
    setScriptList(list: any) {
      this.scriptList = list
    },
    setOrderFieldList(list: any) {
      this.orderFieldList = list
    },
    setUsedFieldList(list: any) {
      this.usedFieldList = list
    },
  }
})


export const useVariableExcelChangeStore = defineStore('variableExcelChangeStore', {
  state: (): ExcelChangeStore => ({
    allMapConfigs: {}, // 缓存当前这条数据的所有映射关系
    scriptList: [], // 脚本
    orderFieldList: [], // 订单字段
    usedFieldList: [], // 当前编辑column使用过的映射字段
    filterMapField: [] // 变量映射过滤字段
  }),
  getters: {
    getAllMapConfigs(): Recordable {
      return this.allMapConfigs
    },
    getScriptList(): any {
      return this.scriptList
    },
    getEffectConfigs(): any[] {
      // 获取已经使用的config，用于禁用已经选择过的映射字段
      const currenttitle = this.usedFieldList[0]?.title || ''
      const res: any = [...this.usedFieldList]
      for (const key in this.allMapConfigs) {
        if (key != currenttitle) {
          if (this.allMapConfigs[key].filedConfs.length) {
            res.push(...this.allMapConfigs[key].filedConfs.filter((item) => item.variable))
          }
        }
      }
      return res.map((item) => item.mapField)
    },
    getAllFields(): any {
      return this.orderFieldList.map((item) => {
        this.getEffectConfigs.includes(item.filedName)
          ? (item.disabled = true)
          : (item.disabled = false)
        return item
      })
    },
    getSpecList(): any {
      return this.specList
    },
    getFilterMapFieldsList(): any {
      // 是否禁用
      return this.filterMapField
    },
    getFilterMapFieldsFlatList(): any {
      return this.filterMapField?.reduce((pre, cur) => {
        pre.push(cur.filedPojos)
        return pre
    }, [])
    }
  },
  actions: {
    setAllMapConfigs(config: any) {
      this.allMapConfigs = config
    },
    setSingleFieldMap(config: any) {
      this.allMapConfigs[config.title] = config
    },
    setScriptList(list: any) {
      this.scriptList = list
    },
    setOrderFieldList(list: any) {
      this.orderFieldList = list
    },
    setUsedFieldList(list: any) {
      this.usedFieldList = list
    },
    setSpecList(list: any) {
      this.specList = list
    },
    setFilterMapField(list: any) {
      this.filterMapField = list
    }
  }
})